<?php
/**
 * @package    Fields - WT Layout select
 * @version    1.0.0
 * @Author     Sergey Tolkachyov, https://web-tolk.ru
 * @copyright  Copyright (C) 2026 Sergey Tolkachyov
 * @license    GNU/GPL http://www.gnu.org/licenses/gpl-3.0.html
 * @since  v.1.0.0
 */

declare(strict_types=1);

namespace Joomla\Plugin\Fields\Wtlayoutselect\Extension;

use DOMElement;
use Joomla\CMS\Event\CustomFields\BeforePrepareFieldEvent;
use Joomla\CMS\Form\Form;
use Joomla\Component\Fields\Administrator\Plugin\FieldsPlugin;
use Joomla\Event\SubscriberInterface;
use Joomla\Plugin\Fields\Wtlayoutselect\Support\LayoutPathHelper;
use Joomla\Registry\Registry;

use function defined;

defined('_JEXEC') or die;

final class Wtlayoutselect extends FieldsPlugin implements SubscriberInterface
{
	/**
	 * @var string
 * @since  v.1.0.0
	 */
	protected $type = 'wtlayoutselect';

	/**
	 * Returns an array of events this subscriber listens to.
	 *
	 * @return  array<string, string>
 * @since  v.1.0.0
	 */
	public static function getSubscribedEvents(): array
	{
		return array_merge(parent::getSubscribedEvents(), [
			'onCustomFieldsBeforePrepareField' => 'beforePrepareField',
		]);
	}

	/**
	 * Adds field-specific attributes to the value form node.
	 *
	 * @param   object       $field   The custom field object.
	 * @param   DOMElement  $parent  The parent form XML element.
	 * @param   Form         $form    The active form instance.
	 *
	 * @return  DOMElement|bool
 * @since  v.1.0.0
	 */
	public function onCustomFieldsPrepareDom($field, DOMElement $parent, Form $form)
	{
		$fieldNode = parent::onCustomFieldsPrepareDom($field, $parent, $form);

		if (!$fieldNode instanceof DOMElement)
		{
			return $fieldNode;
		}

		$fieldNode->setAttribute('addfieldprefix', 'Joomla\\Plugin\\Fields\\Wtlayoutselect\\Field');
		$fieldNode->setAttribute('folders', implode(',', $this->extractFolders($field)));

		return $fieldNode;
	}

	/**
	 * Normalizes folders definition from custom field params.
	 *
	 * @param   object  $field  The custom field object.
	 *
	 * @return  array<int, string>
 * @since  v.1.0.0
	 */
	private function extractFolders(object $field): array
	{
		$fieldParams = $field->fieldparams ?? null;

		if ($fieldParams instanceof Registry)
		{
			$folders = $fieldParams->get('folders', []);

			return $this->normalizeFolderValues($folders);
		}

		if (is_string($fieldParams) && !empty($fieldParams))
		{
			$registry = new Registry($fieldParams);

			return $this->normalizeFolderValues($registry->get('folders', []));
		}

		if (is_array($fieldParams) || is_object($fieldParams))
		{
			$registry = new Registry((array) $fieldParams);

			return $this->normalizeFolderValues($registry->get('folders', []));
		}

		return [];
	}

	/**
	 * @param   mixed  $folders  Raw folders value from field params.
	 *
	 * @return  array<int, string>
 * @since  v.1.0.0
	 */
	private function normalizeFolderValues(mixed $folders): array
	{
		if (is_string($folders))
		{
			$folders = [$folders];
		}
		elseif (is_object($folders))
		{
			$folders = (array) $folders;
		}
		elseif (!is_array($folders))
		{
			return [];
		}

		$normalized = [];

		foreach ($folders as $folder)
		{
			if (is_object($folder))
			{
				$folder = (array) $folder;
			}

			if (is_array($folder))
			{
				$folder = $folder['path'] ?? $folder['folder'] ?? $folder[0] ?? '';
			}

			if (!is_string($folder))
			{
				continue;
			}

			$value = trim($folder);

			if (empty($value))
			{
				continue;
			}

			$normalizedValue = LayoutPathHelper::normalizeFolderPath($value);

			if (empty($normalizedValue))
			{
				continue;
			}

			$normalized[] = $normalizedValue;
		}

		return array_values(array_unique($normalized));
	}

	/**
	 * Converts stored raw JSON into a computed dot-separated field value.
	 *
	 * @param   BeforePrepareFieldEvent  $event  The event object.
	 *
	 * @return  void
 * @since  v.1.0.0
	 */
	public function beforePrepareField(BeforePrepareFieldEvent $event): void
	{
		$field = $event->getField();

		if (!$this->isTypeSupported($field->type))
		{
			return;
		}

		if (!isset($field->rawvalue) || !is_string($field->rawvalue) || empty(trim($field->rawvalue)))
		{
			return;
		}

		$dotPath = $this->resolveDotPathFromRawValue($field->rawvalue);
		$dotPath = $this->normalizeLayoutIdForRender($dotPath);

		if (empty($dotPath))
		{
			return;
		}

		$field->value = $dotPath;
	}

	/**
	 * Prepares rendered field output and ensures subform rendering gets normalized value.
	 *
	 * In subform render pipeline Joomla triggers only onCustomFieldsPrepareField for subfields,
	 * so we normalize raw JSON value here as a fallback.
	 *
	 * @param   string    $context  The context.
	 * @param   \stdClass $item     The item.
	 * @param   \stdClass $field    The field.
	 *
	 * @return  ?string
 * @since  v.1.0.0
	 */
	public function onCustomFieldsPrepareField($context, $item, $field)
	{
		if (!$this->isTypeSupported($field->type))
		{
			return '';
		}

		if (isset($field->value) && is_string($field->value) && !empty(trim($field->value)))
		{
			$dotPath = $this->resolveDotPathFromRawValue($field->value);
			$dotPath = $this->normalizeLayoutIdForRender($dotPath);

			if (!empty($dotPath))
			{
				$field->value = $dotPath;
			}
		}

		return parent::onCustomFieldsPrepareField($context, $item, $field);
	}

	/**
	 * Resolves runtime value from stored raw value.
	 *
	 * @param   string  $rawValue  Stored field value.
	 *
	 * @return  string
 * @since  v.1.0.0
	 */
	private function resolveDotPathFromRawValue(string $rawValue): string
	{
		$rawValue = trim($rawValue);

		if (empty($rawValue))
		{
			return '';
		}

		$decoded = json_decode($rawValue, true);

		if (is_array($decoded))
		{
			$basePath = $decoded['basePath'] ?? $decoded['base_path'] ?? '';
			$layout   = $decoded['layout'] ?? '';

			if (is_string($basePath) && is_string($layout))
			{
				$basePath = LayoutPathHelper::normalizeFolderPath($basePath);
				$layout   = LayoutPathHelper::normalizeLayoutName($layout);

				return LayoutPathHelper::composeDotPath($basePath, $layout);
			}

			if (isset($decoded['value']) && is_string($decoded['value']))
			{
				return LayoutPathHelper::normalizeLegacyDotPath($decoded['value']);
			}
		}

		return LayoutPathHelper::normalizeLegacyDotPath($rawValue);
	}

	/**
	 * Normalizes computed layout id for LayoutHelper::render().
	 *
	 * LayoutHelper already searches under JPATH_ROOT . '/layouts', so leading
	 * "layouts." segment must be removed.
	 *
	 * @param   string  $layoutId  Computed dot-separated layout id.
	 *
	 * @return  string
 * @since  v.1.0.0
	 */
	private function normalizeLayoutIdForRender(string $layoutId): string
	{
		$layoutId = trim($layoutId, '.');

		if (empty($layoutId))
		{
			return '';
		}

		if ($layoutId === 'layouts')
		{
			return '';
		}

		if (str_starts_with($layoutId, 'layouts.'))
		{
			return substr($layoutId, strlen('layouts.'));
		}

		return $layoutId;
	}
}

