## Subresource Integrity

If you are loading Highlight.js via CDN you may wish to use [Subresource Integrity](https://developer.mozilla.org/en-US/docs/Web/Security/Subresource_Integrity) to guarantee that you are using a legimitate build of the library.

To do this you simply need to add the `integrity` attribute for each JavaScript file you download via CDN. These digests are used by the browser to confirm the files downloaded have not been modified.

```html
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.9.0/highlight.min.js"
  integrity="sha384-2joJCG6VxpGDyBS6SI7qdH4d4GeplLs48X3WOFoh28XvAE6FOi4V3gCEsM3Rzwi5"></script>
<!-- including any other grammars you might need to load -->
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.9.0/languages/go.min.js"
  integrity="sha384-WmGkHEmwSI19EhTfO1nrSk3RziUQKRWg3vO0Ur3VYZjWvJRdRnX4/scQg+S2w1fI"></script>
```

The full list of digests for every file can be found below.

### Digests

```
sha384-2joJCG6VxpGDyBS6SI7qdH4d4GeplLs48X3WOFoh28XvAE6FOi4V3gCEsM3Rzwi5 highlight.min.js
sha384-NTEljp9xpubNSGgMHaDURvCx1HIRNG2RM7s9GHTFAZJs5/0GsFYpIwpHG5+rwH8g highlight.js
sha384-nhKFTE0p1v7w4VPdiCXEuAKBuGnZRjVarChm69kITyFj/Dfcu1U9oxHOtcCDQVWe es/highlight.min.js
sha384-SGjBfRNAr26l1YCG6rvyyYXGJ4eu7FbjJrtvJrPMHn+O+jKL0WQgT2clE6wbTsCM es/highlight.js
sha384-nhKFTE0p1v7w4VPdiCXEuAKBuGnZRjVarChm69kITyFj/Dfcu1U9oxHOtcCDQVWe es/core.min.js
sha384-SGjBfRNAr26l1YCG6rvyyYXGJ4eu7FbjJrtvJrPMHn+O+jKL0WQgT2clE6wbTsCM es/core.js
sha384-BF61I4vqlKI+Z6dQUyVuqQEE9RSYZWptePUuYrYLfxlYJT6r9qNemzfyAqJIXHRJ languages/accesslog.min.js
sha384-ocQrCZO2M6Koq2Yw3/C032omZPrKkBqqc47DWboypId4OQXaKJ0zAC59YDWlx0PE languages/1c.min.js
sha384-CMT+D14J35o3MzFS0PS+C2Uxx7NsC/9LDa20e/cSnYPEuKGR9IaO/Y0E5pN2qeFJ languages/abnf.min.js
sha384-nbszKU69mKbDXQeeKtVe+xu1QWXenOiWYLzCRdSCXXZiyo7+Sc9UdUjiikNxG8ch languages/ada.min.js
sha384-SXlbgREKQvT6vnCiz9PJKTyS3FHFLPIj0EKXiMUEp1Xna6/gOL88RRVo/AsE0mR/ languages/angelscript.min.js
sha384-bvrESIqIr21pkx86VgiN+Vfq8sRk/6xTxMfbo0PDENLfqYr123El7PUCzqJmir4P languages/apache.min.js
sha384-FNkevfMdwlt5PY/joajGDoOOJVgElZtmde2301WYvRzYu/yhe/28OLBDxs3JsITM languages/actionscript.min.js
sha384-DXh21OyhY78ycqaPwsochxlyc+9KtEG+v6BZxyDRuXWvPXyNvXi2wQKfdaZKlEyg languages/applescript.min.js
sha384-CbMYLaAbGE/EFV4eR81PkSRDh4G3ewUxVciKGei7MAyQIrGv7nh9BwolkAmLBeL3 languages/arcade.min.js
sha384-sf6kQzw438VELcTHWkS4YdEyIzCPQbGZLke4v5rtAIBrnGsuvCILPWGeIcDIVLmX languages/asciidoc.min.js
sha384-60YzKSwA14gAvQbuEe+XjCv2EepzDhDjELBciyGY30m9yD6zdxPDVakewhURpuJY languages/armasm.min.js
sha384-0thj2cy99KLUBB/ftYmPYVhMdSjNZZd3hG/FnuWfshVDkQTvgWHF+n+PdQ7zwuW3 languages/aspectj.min.js
sha384-JSatX9SPqm0lqZqilowpQUi4tqiqhgyObFqm0KvWkMZ2tuCpcHoxgmuKVFCSCbDt languages/autohotkey.min.js
sha384-BK6bQW09cPx3jR/N3JQ4FnK2LI9Y3b3zrDosiReW5PlhzemA2ZPeOReQVk4Btsai languages/autoit.min.js
sha384-2kSvQKl6UWC4J0b6x8BlBWmTtvRIe+Son7FiVcKXoQYYuzcIbO5GbLz2XRbHmIkE languages/avrasm.min.js
sha384-JthKd/4xod1HnNOcxLd1CZo4VWSuUNLTIgODIlSdtCWCyBPA41+jJTyEtC5fjyFp languages/axapta.min.js
sha384-OqfnIfpOmAlErKRs2a525ZcwLuZ8Z1x8hPdIwBcMtHkJBbHnpJ6wX5/LVzASHkD5 languages/bash.min.js
sha384-KfOKKkbUp6gWmUdMPNCj5Hqqmf9Z03BTs0GnM3hyrYlNBYXSt4O527wBkKrgxiIl languages/basic.min.js
sha384-79U3FFsfVk8mFdeTuAdp0bLlLWAZ2Mh3zYTr0ndb1luM3RGpgFFiyInsDU4p4VqC languages/bnf.min.js
sha384-+kQeWvlXeWctdKQQ8iGrG1JdmiKRJHAqofS2vKNnJ24H25HqMhb9kIQC3NYrsoch languages/awk.min.js
sha384-LKuLCELGFBBi0CScpYRqq6/GCP8KEnoSQ/ZJ9WRnur6+5XdZd+ajgrwd1G2eN42h languages/brainfuck.min.js
sha384-pQozISLWg9CjdEOelWIGsMOoVC9wA54X9kXSH22Eg/nj4WUCU0l8CvA0GH1hhxie languages/cal.min.js
sha384-mWenE7UXKnmYRcJ3mh+Os3iZ43BmFf9x3AZMM6gi/2sT6vxouPWspbTdREwWDO3w languages/c.min.js
sha384-X9ejkQYTr8UwtUbCNj5TNFyggrCb1iV8lydyf6Dz1OyGJRuVopefQTJOK1KevmS2 languages/capnproto.min.js
sha384-nVcY48hEwT1Yo5Zs4U9fyUDtJODFWa0egIHbhksE9ftE8vxEs7dv8VvniI24PMU1 languages/ceylon.min.js
sha384-YuLT3q/Co5XpnHSrXi+r14qWIfXfFw4yd88JywxmeJyorSwjjzEo7wMPfrd8tdKp languages/clean.min.js
sha384-01/jwK07TwTjPcx55RJ1V03zqCdkEK8QBmgLSbcgcyl1uZ1AwjuSmufE481pF0nY languages/clojure.min.js
sha384-JHMQSU8sCMCdL753bS07Oioz0pIyBJnH5TRJy2QIQqDPil37Ezn0fO8LBx+F6/Mn languages/clojure-repl.min.js
sha384-iqJTnKrAPlpoueHQticEO1SgUDmD+Lv1EGplrE41YfNgnnVCt6pLeVjutOnr/Ddi languages/cmake.min.js
sha384-Djexk3CUuWlhu8etDgJrIqYyX/2iPKHlyqZo1l0uTRocpX+lFlDpNq4J8YaLd85M languages/coq.min.js
sha384-pF0kkJHJ7iLK/GTjymCseh94/WcMBVV6C9lKk2HdA+dut+ACEizCpQbrzWE4fpfv languages/cpp.min.js
sha384-Gu6/qq0M19/b9qnSKY5N2TpxykfnuFQlprawsvsy9Ixcu55tvT+BsOtscAcdnpSd languages/cos.min.js
sha384-zN7rKt9V1OgNoMrsbrSGe1x9oL1N+6SczPLqhi2vRQNealAEGZBqROnVqMnc5FKL languages/crmsh.min.js
sha384-rw0NhtB8AIhqA9Pwqh0Vt2kM/HuvVUBgzij3aTww0NIp6g2RfKHYlE39Vg2pxYSV languages/crystal.min.js
sha384-wWP4JQEhRVshehTP7lUMDn3yhDI2+398vN2QW5LBt1xIpK0Gfu4dPviO8tP9XRo5 languages/csharp.min.js
sha384-3RA6jlKLtef98fWpGKtFWwqUh3RIT+Fg73EzHy87JjmQZLPfeQvcOMoOpTwOhRiI languages/csp.min.js
sha384-Tdk10VpaCEmOnsEK9S59mic0TQmxPJT9c54Rfcarm+FxLB/TePlvsdj8PXmXZPyO languages/d.min.js
sha384-9N7m4GS3gPm6zFPGhkir0Vn61SBcmTq2ZIlgNo6Jm3N2khSHxFvQ9rJ5WRfX5WlN languages/dart.min.js
sha384-ii437zeP5iV+Dk6oU6OW83sd8HK1skKMk4zkamqKUfp/mSRFeK26M3IokGhAgTpS languages/delphi.min.js
sha384-yE0kUv4eZRBzwnxh0XmeqImGvOLywSpOZq2m0IH40+vBtkjS716NYEZyoacZVEra languages/diff.min.js
sha384-AqTOESQu37Lj9i0LQjA1B5Ju2XOJzwB7RR1NYcpo+JgnUF+UTdQi8km+UzU8uYBZ languages/django.min.js
sha384-ThDmUDUbkLHd3YjqnAKK6DGmCqm/zinzT9ezHFHk+GPnKaM0dvEM3KfFeoQUt8KH languages/dns.min.js
sha384-6+/BHm9a7L7KFZE/myAKBddEjwfPovVtBzDrEu/6VqocfSqvRMyM+Lf3k9n89yGH languages/dos.min.js
sha384-H3wUmCndJNBOuIIzcLSr2zLLZYs8PYuw6qKyENk7THEnYQJvOMDBu42P9XsJSRKw languages/dockerfile.min.js
sha384-KAqJVHyldYmpO+CZL4qvQxr1GDHXYdvoNtMcN8IouYTVaq5ZiHp71eAjGFLUWl8M languages/dsconfig.min.js
sha384-Ds5adkjVRA50t79I03B3O1Nf2RpYTYQ0m6arqivGOTJMiMffJyF+Jf4CmEil9LP9 languages/dts.min.js
sha384-znWzQaFxBzfexHijTcI4LhThosfF573Udkgbo96l8XksZPbRj69/m12lsh5gadft languages/dust.min.js
sha384-LT6ea03Ihj20gYuU/bfouACcd8rNgZAohWItMlIwiS8CUEIBt5BEeVBZj/Xzngp/ languages/ebnf.min.js
sha384-IKAy8uG+2GJ7OKT8sMCyI+irjkHsROGLhUVfVtbTXwtjibq81wfzBddeahMGbprC languages/elm.min.js
sha384-qMYXRUi/Q9Ukntag3SOX3rOfkgLmmjlGA84X26QeOGixvWyfSGEiKwaw5ZKoSyBH languages/elixir.min.js
sha384-HH6ifvxwyuXXyJkqfZjOkyD/ZQIXMe3zZkjUKZKwCDcpbiiq+jLNyBEclXTBwK69 languages/erb.min.js
sha384-be/OrUAzl+Gj44muuMsL6TDJGNY0GPTXm0tNmStW12RI78CcxFIIqeO9P+3VXefw languages/erlang-repl.min.js
sha384-yZpyX3GaZ2n2wTQboLuT4pMbAyEK+icy793eW60VdZdVZMw5g5jxArbZtFkh+6ri languages/excel.min.js
sha384-u+PhAfFfNA5rLKSQCIUEwpgEcjshCgJUI4bjX1PZBZ49MYPZnEV+p5IXis0L4Olt languages/fix.min.js
sha384-hsThqIkkHew8Pfgbao+v2vLaYUKOUiN8CiPSkJRrmxTe4LcgVjZLC3Rz4z2zMT+X languages/erlang.min.js
sha384-WvivEbuG+ynOu3BbbceJknlKXBqe8Va2Zo7+TgCqtErB61cfboeQHQOyMapYhkUW languages/fortran.min.js
sha384-AxlyEB2p5SFFPIJV0Avp0RR3ld8cEW2H5ktrduSFHpO8ytXE3Ch/k6k2zUVrp/Ih languages/flix.min.js
sha384-5dn3BjT7CfwjlPvSgpWYsjRBeZm7OfUw6n5glAVzYz0mwe9AHS1gxPQX3ezqZLje languages/gams.min.js
sha384-BCxqi+Xcm31Y1fr9pq1kelLJM4YdFXCRRIBQgY9d7/rE4pZwXuO5CXiHaZx/EHK3 languages/gauss.min.js
sha384-OajfDlneoFR91p4CwU1HXDF8d66Tz+M5REwKIak83BHekUXi5T1eo/tQrJR+UbRD languages/gcode.min.js
sha384-Xc0i+mMsiuTC4gGqr36cSIjIizpELF0WcTmluioQHobSn2zQo8+LdD5dPujiGDlf languages/gherkin.min.js
sha384-PkNQzkPieB8PCovDH3VAfEBO1RvFP6kPVFxLkO2soZ3MoAaf/TeKffj2vMLGxJlT languages/glsl.min.js
sha384-kBAfOJR5u7K4rRCe6XBgmGfW0P+CxzQoUjt2HiIDym3zmASPuDPObmtAXxDG+2Lz languages/golo.min.js
sha384-WmGkHEmwSI19EhTfO1nrSk3RziUQKRWg3vO0Ur3VYZjWvJRdRnX4/scQg+S2w1fI languages/go.min.js
sha384-yqNr+JW52pR9ZNw8CIHWgGMIrqhKhuOBDSlQ1ulf4Gt6wqi+VUMGHP5j6ReSSRY5 languages/graphql.min.js
sha384-qD3OsS5noBhrPG9j+xbJa4SZBjPyb1Va3D5b9ZApFDVMXowDQ/W9D5t6A1z7Urkg languages/gml.min.js
sha384-6TGxjdrZB3xEK6UL4EcWt1IQb2v4GpvxFzOBntkuOlDC688Rf5gqq5XTWJzO1izc languages/gradle.min.js
sha384-N2BpqybE0W3/jhUPUCcAr9I33RlAVfy/E+lEPCSY+NLvXJdrOUWK5+/ocIoPmA64 languages/groovy.min.js
sha384-gwE6A5G38G4NqHANd+YB0huqAVDjHePFwvBBBBKU7miH6mkz1RNWArUIqpMpK2oD languages/haml.min.js
sha384-PqtEiFXspmwbGzxcwaN07nChfbxIAJHK4bTNOnytspmN9Q+LkL3AjxV5SgUI0WJQ languages/handlebars.min.js
sha384-EnLqy/LmA2yShypXJIP5cvUz0S3jJ5l3PnZcMRE6JQdpeNOttU97yyQAsdTusDbl languages/haskell.min.js
sha384-uUmo8PWFx8gz1l7YQJJni6Enst2ZpJpYOnegvyrILp5YdqT9dwQuStgIob3F5few languages/haxe.min.js
sha384-wKQ/Ehi5iZ3saF4TYpASquMJHrJQ03Nk1eUKmJhCT4LBoHxWwHGnPjR73VFlnS6E languages/hsp.min.js
sha384-wt2eEhJoUmjz8wmTq0k9WvI19Pumi9/h7B87i0wc7QMYwnCJ0dcuyfcYo/ui1M6t languages/http.min.js
sha384-U4QwKZUBikMarf4tvg4pvE08xiNYACuJuSezyjTby3pDRF0J6c9Rc2nnEQL4j5SL languages/inform7.min.js
sha384-rnI8uyY4tKvgWZzwMpsIPKn9uxag/vo83pmbjixyhE7aggFOyJy4bQ5UAdIXj1Pp languages/hy.min.js
sha384-QrHbXsWtJOiJjnLPKgutUfoIrj34yz0+JKPw4CFIDImvaTDQ/wxYyEz/zB3639vM languages/ini.min.js
sha384-ziJ6FsBY7nWvis6IJnekWGpFTnxibSusxDLDGSwMrFcTH7cD+rDbUDIyS5SKHL/N languages/isbl.min.js
sha384-rA0a8MPnk+1v02O2KuIgAZKtVfWdW+6pG0aXGz7rJnbhjeDBkjd9lHin3XZABuQ7 languages/irpf90.min.js
sha384-l/9Uw2IKYxqWShUl/tjB8ntQuuyvxl+Y6SjEVbTZPuDxgQBWWFk3/L+a9Xq3kdRv languages/jboss-cli.min.js
sha384-RbRhXcXx5VHUdUaC5R0oV+XBXA5GhkaVCUzK8xN19K3FmtWSHyGVgulK92XnhBsI languages/json.min.js
sha384-fyOrUSw80R//DSvkLRfpGAxhjx6MpL1QoWcTCG9hXXT74sh7gXeUrstn7IwT0G3K languages/julia-repl.min.js
sha384-ZqjopgKriSJBeG1uYhjsw3GyqKRlsBIGaR55EzUgK9wOsFdbB67p+I0Lu9qqDf2j languages/julia.min.js
sha384-ar1PV/gV1rM0g1bTSeJG9pFUGOuOsLuU54Fy1SaNvISa/KP56jGoyV+EHQS6NFQA languages/lasso.min.js
sha384-6K5LosLSpBmsgch8WDi4VAu8AFGGqFHLbaj8VyhpemdlB1u4nq5f9JMVU7mECgFN languages/ldif.min.js
sha384-i3cXfx31B+tXGBPZza9xHYTwPuoURF7qCWk9ui1X43UqlJI41EqRRLQ8RBnvtHYL languages/leaf.min.js
sha384-0ZgcpgIDZoDPMdyG1FHnl/MByMKt+W52UltboQpuA52+f4pniKOMFXfIfe4K4Byc languages/latex.min.js
sha384-4m6lD7Kjp+jLCovQroP3YvhgxUBihj/I11CBCi258dx8RYXe3u0zqt8DOrsZYSTr languages/livecodeserver.min.js
sha384-+LHHMbAXOUlvquvrQZ9LW4KeR2nwcsh/lpp7xrWu7KuaDSGgAYBIdm8qCw97I1tq languages/lisp.min.js
sha384-Ss0h15uPWBGpCpSOjtNN1Q38nRKRLiFJkKxU1q+R7WP+1ydm5V7hR/J1ED95tR6z languages/llvm.min.js
sha384-T04Yu4dcDCykCMf4EbZ62u3nURYEVkpphRGFhF/cMu+NrtDqoRHgbVOZz7hHdcaO languages/lua.min.js
sha384-9+oedk3gx2l9P3spEsJzaq06+1ETUMlQ2w7eXJ3lWAQMuaoBMPd2FZK4v2BlUhgb languages/lsl.min.js
sha384-NIrob3StFQyD/nlOsXVCeRsJ0N2SvFEDjFtYS393wbD3CY1eT+2kwT4RL7tpMMhs languages/makefile.min.js
sha384-KveG8st4Ls4iaD1XzpsBzVc7g2K6kxnbxlZOy9cH6Knla0ZH9jsloP8nOOs6WYMP languages/matlab.min.js
sha384-5yH96HtcThJqsdCBlhqHsf+le6u9nRPuvSQUjjr4Sg7pfPmhT49agY6EzJ/pFpCK languages/maxima.min.js
sha384-mCUujHHbWJEjcupTTfWOk9YR3YCYNHaA578+TTXUd4LPi7fGNuMQbysbl1pmcIGd languages/markdown.min.js
sha384-y3jkby0nV9TXHdpLnC8hL1gbtm8ud96pgPj3H5VA730hCYOS9ePWXnoeEnoh8aAl languages/mel.min.js
sha384-eSQHi1xPeYGHNR227fIyKTqzjU7Xn7RQ2TtnJ5OLN7Hoa+eCJS3WU87ebHYahSVq languages/mercury.min.js
sha384-nPxUWN1vhyfh/b948KmzsJ9iO8h04gR6NpsBSHAh4i5KdAXXrdaeRGdGMK4Cq2Ao languages/mizar.min.js
sha384-Au9YnxxJzBIPDFhCGzyfVezXn2M+TJ4IkKGZQtKIyUqfXq06rjNp6+zxtjjPG9tL languages/monkey.min.js
sha384-zo4QXIM5F0f6KYdnuY6s02e62G7rQUQrc9jhM6L/PZWX0emwzVRLIMUFh4hTZrg0 languages/mojolicious.min.js
sha384-JuwM4QgALf4/n2PaHzRCU3ppICcAenDKOyhbrthq99UkLQOkEVdSf0Eig9g64vNI languages/mipsasm.min.js
sha384-k9oOOoT0EvJI9jkMXeSWCfVmFVQwVV77SXLhQSlO4PEmIGxsk3r9MaFR+i9T3eY+ languages/n1ql.min.js
sha384-IAS9FaYc2016cJCQlx9CxX0bu9NbMWXML2MgfcAhFsJnFzsM4Wd83i2bPspwhxXc languages/moonscript.min.js
sha384-p6UV3HZc7zs9wUF6j4Knk3bpx1/JnhlwDoFnKJAOogB1cUi32u31NlnZ2s2dW8IR languages/nginx.min.js
sha384-vqNCsfwdY8G3LMt7gIKLOVi51hjD506kckOiKh8fF3aP6JefnON0lvvEOH7GZPne languages/nestedtext.min.js
sha384-aDGSumJzWocIW8uUlSvYbRE1BwKU319shHv1p8V6DsT5RNsImzghku0TxI+MrU1E languages/nim.min.js
sha384-Cu93ZX4MeViuo8zGfs3i/Rbyj81UUFg0DLBKhn63FlLi/qoH39kOO++xzdoUJq/Y languages/nix.min.js
sha384-xPT/DjMX+BQlJVnPobXtRQHno6nxWndjO33v2Xo4s1QO3BFFKTZHPvoZEfMzFzAD languages/node-repl.min.js
sha384-laCnSBtpGFl9ww6pMsbiDiJocbu+HlV9qgfWjOoArDqQrnlxH3Qm9kia5N2cicKM languages/objectivec.min.js
sha384-WhlOQ8tkNAn2N+yTa7njealQ8ue+22g0Zw/iR6A9QtSz0XQZzBebXuAvINaGCpdY languages/ocaml.min.js
sha384-4brKmHEMDX8vKcYWEmkTz78yMmB2BIsCRMklhhaGCYwfZu/ek+VWCWvxT7Ghtv6q languages/openscad.min.js
sha384-PnZtWLQ1VIP7reVQSB++EOExTmFJo4E2ar6uVI3m9z+LxtgBYxujhyPUSw97np11 languages/oxygene.min.js
sha384-Un1K0gr5jJKP+lLTAhj6Wvyjo5x7JgwEmajWo8JXMU8Fj7GazpSrFRzW/CWL5DcE languages/parser3.min.js
sha384-0KVrckZbdy1irSnLt4c2/2ChyACCx4gdBhrjwpzfszqDweaffQIuq0TBiYO+MpJ8 languages/perl.min.js
sha384-Oo+KBAbqcXAtR0/yFlZsRvJSAITfvxhJ761p5yOO4pyEtYO6n/5j6eVI1dG8aGct languages/pgsql.min.js
sha384-qZcRyhSQHLwNANrkp9RM43BAlw9rKj7SnAN98sGmMEADm/8BMVSJo/rRiN06uY6L languages/pf.min.js
sha384-i5dGKRLoml9823i21v5ltbNFQCB7OAwkUve4TaJrwSnXRQ1kOjT+Wa8jX/EOj434 languages/php-template.min.js
sha384-c1RNlWYUPEU/QhgCUumvQSdSFaq+yFhv3VfGTG/OTh8oirAi/Jnp6XbnqOLePgjg languages/php.min.js
sha384-Hu1K5IR3tWjqNzVwTlOnFre8rB9qJ4qjSVzWmBVKbaTtCSYxFRPZrlRHb0chbeZK languages/processing.min.js
sha384-ofjxHpechXkaeQipogSyH62tQNaqAwQiuUHOVi4BGFsX5/KectIoxz16f8J/P5U0 languages/plaintext.min.js
sha384-LWJZQx0dgGhEK7snfNYrQs5K+QKD1sOmE02sOQCz4br9UmqSJDvPLoUVFaUyFnjq languages/powershell.min.js
sha384-H1KFsw9FjuiPd6/Aa/qsyqoT2Gk6TcfJhf0oaLD1x+a8aJ2GqxxA9oKDhQyh0BqC languages/pony.min.js
sha384-UM9Lhdjkj0JpmTGkTvhc3JrFvbPmcGLGqzLsKAMP4hAKxussecmapCxDpWC5fHFu languages/prolog.min.js
sha384-7wPGltVO/JeXBh24riKqUEDrRGzBtEoJAIxLFV7EnTtBzHRpWXv1VhUfuAgW3tVQ languages/profile.min.js
sha384-1ALwZ77/iqBsHTR04APuiMf4wWnFgY4b8VUXQHCjPfywk7S9K76uwvs1xLZUTfY5 languages/protobuf.min.js
sha384-tlJaeulTOJz9iqadCPTY2JwJVayIS3pMaQkEOgibHjwjmI6Fve8TLKvsR8tfmKwa languages/properties.min.js
sha384-0fGNJrsJnJM+bCoqEWD8wAjBnTVkiiieYzUytYZ2EjwMDmyp+bky2QVw4w5QmqZZ languages/purebasic.min.js
sha384-fGC9yhxS2ALC1B7VH3QJtJB74sL8L9HGTepwFeHWnStwNBaEG76aCmL4Ll/dG3r8 languages/puppet.min.js
sha384-bKSBGjy2/8jltjSAlvCjAvEvjy8osMZfj37sBThcXS7n6SppraMRghmLmAxiapyN languages/python-repl.min.js
sha384-tySsyY9P8DFEhItXhGoA09Abo0SUBjuQjXGcCRrHnG1Npi0+CJRwOeQsJgEoPz2t languages/q.min.js
sha384-FWJTgPmIGm1+zpNhubuHRC/ulS1UK7hAZ7qiUUmKD8yGfPcn5ZXYBd2qRgi6L8Tu languages/python.min.js
sha384-3yfE/mGDflH6NuUjuQX9xUlrOGH2XJhnt9GD2Vr391/MpeZuJY+Cg1gxO/zoUYDr languages/qml.min.js
sha384-+z7INDMj7HMs67TuLfoyKa6TUHdTweahANtJ1Spg6uCa8/f1jFUf238jj53TDxmP languages/r.min.js
sha384-Z9gnV9+ow776+3ev5SPwjO6FLYQSwGZK4N5mRQ0mVHBVkpq98BgOc6o6+S6M0aJH languages/roboconf.min.js
sha384-AklVVUwsER+PmPMYZ3eVQQLqIFOIs9C7yetKShYFHsdq80PXa+MVE5OAGJxO5ktx languages/reasonml.min.js
sha384-6wYRle/j4CxyJLYOVKG5KaBZE0ZLZG1Q3AhBf6mq/+oJPB0wlV4MuLZ4TF1lk8ha languages/rib.min.js
sha384-BTOhup65AUZi6c+k5ogSHtcxJxBcf4Oc912ZjRqtM55k5AyCgv23EZLz9QKpy+BX languages/routeros.min.js
sha384-zas25yfexH2kS3lABsMOlMHpph+Ra3fX5CtzjCfDJdLk4oGPXAsjmswik04J6Ma0 languages/rsl.min.js
sha384-pFZpTUpdH4YEXSenc9hfKZ4uCv2IQoJQCIlIHpA0fM2cvTVH8LuzQMNcGSRGeJG0 languages/ruby.min.js
sha384-MsKzzzJz/hSFJ8KI72fazE48Hx/b4azofkfJj41UxJHSsnatWjYp9KKpLF10stgV languages/ruleslanguage.min.js
sha384-5XEbQCxcp5Sa4OidPFkgGzAEcyJsr2zWRw1uTUxs2oqlzi28N+epAmfmMjQO2bby languages/sas.min.js
sha384-tbWlb460xfE9lNi4sNjNnZ4Btfos1He7bWO+9UmtNq4TwJ7F7nX8gaR0e6sVJsUk languages/scala.min.js
sha384-h29HWMYGysZp5v7HMfLgfj7PMRyj5l0/KqKdXWYxGfcfh4GxKPNGM/Ria90QFqoZ languages/rust.min.js
sha384-wDAOgnGx0t7X9tTKddfqp7kcfAYVDaQDRdfDD+47IukNBVuVGFldV80H4qDTHPOR languages/scheme.min.js
sha384-1EaRaf6maZwM9EBh63r/zfWgwq7OrP73vE3uqdfUE+/j/ElX9b53hhnBYNlNimCM languages/scilab.min.js
sha384-PDEMny3uYpCrSKwqVGLAwqCsc9u+9hYXauxVPqO6RXmlxSNKLoRByEBigU5lahmK languages/shell.min.js
sha384-qlNJsp6TV3mWq9A23lJi7P95j8X7Hw9jhfQ8I63Mo/2aGnQQkeFkcQHfHHLx+uR3 languages/sml.min.js
sha384-0szVjKKATgtBXpYE6ZwLFEaR+msbyW5uBxx4hqaWpPJ0A+yf1P8czY+QxYF9prTH languages/smali.min.js
sha384-z8va2UZcRMQILb9EK445jGCl8ly8Tj6OjHuSJtCn0HfQCxVspFiAAtws0JPS1hAi languages/smalltalk.min.js
sha384-EzqYdGBz7phyxvQ9Dgh9Kj7TxiexZpoZbihWUDSxm/N27XcopY8DCR37ShesAR/W languages/sqf.min.js
sha384-8q00eP+tyV9451aJYD5ML3ftuHKsGnDcezp7EXMEclDg1fZVSoj8O+3VyJTkXmWp languages/sql.min.js
sha384-yHv2Vl/ewwVbtbqSAd9FbE1Tp47JYrMrbUfOIQiODaVw/KQQNX1SXcCZHf8wkyTr languages/stata.min.js
sha384-L+/d4w/89ob7ecvWtGeUgudL+YQxAvzC6hGYtfAhLFAqvAJV+fdDdvPFR5BWunxH languages/stan.min.js
sha384-63nMiVu1JxBjtmzYOi3xVkS0i86BqlaxiyGtmqliNxAiextUZXSR+kmCTYm8tiY2 languages/step21.min.js
sha384-CHwy8Ggza8Qqw7BUWHZR1qiSoQowxPo7/QPs6/Jjw3ExUFzC9KqEP8uYojqb71ED languages/subunit.min.js
sha384-LD0Ql9hUNFGOQfpDUMtgdhoyZax+t3OZyULiQ2bbRATKxhLxTJuUTPI6cClZ+JyI languages/taggerscript.min.js
sha384-Hj1P00F6D1x30YUcm0DmnMX982aziC3TIuzshEf+iXAU3lRAx8lHaf/i0tKzRO5M languages/tap.min.js
sha384-88ARtX8chYVIZZ4A0cFUpXilu4qORXSqH3yTVPOSoOSluZvBRZ6TAzO22TvW9Kg9 languages/tcl.min.js
sha384-gIM8FskL2T3Ril4K62H6ZkAwlLrVyzOfYSLX6weXIremtGOgJhKnkYJS2F+zAtdM languages/thrift.min.js
sha384-xsvxj8MIukMHSGjcnthzkIqrO4dXTKOhgzWoRwGuMjJqs9o3L7V9TFEolnDm7sr7 languages/tp.min.js
sha384-u6Dcpci4KW4Z5xGImPYwgCCZJ4EhI2THZLanipuEfVkRnbv7S7wVZAQ2/6L2Q0r1 languages/vala.min.js
sha384-+o+wt9LKrJr093fuXe59myi53sotEZ0z9+zFloHncK3RudScyrwBfGFj8oW5urR+ languages/twig.min.js
sha384-+y0KLxbRrWxqnfGRhWWQTHHEwDd1OhkOacgf5QfJa+5ydoCf3SWObb+XrzxBSfqa languages/vbnet.min.js
sha384-Q2pUgWvb2tvXq+8kfdAdSGQFgjcgBh46AFq4OzvGQTDptWTry+4Y9erDzpdag6vi languages/vbscript-html.min.js
sha384-51jdo37NeOkYYUHq3Pv0sCaoGJOAFt9I2CIGGoQKDgY7YnquSMoA6f7FjfkxfZbb languages/vbscript.min.js
sha384-eyRqKxDP84fgtH6HUtm9/CDSUuw5UTFOBA8lTuZUUDougLHHjOswaCbiWH8ODnRV languages/vhdl.min.js
sha384-9xU7pqpsXCcKkzy15LLkblwvB0dkTn7qzmdcZ/X+gjiHaPsqdlAXM3faWbekiSGC languages/verilog.min.js
sha384-WcP1nsZR09ttNCYSUJGb4LmXw+/kA1O2iyojERHIrTB7Ou0ONLTsIkFOnFF2Umkj languages/vim.min.js
sha384-fWMR10q8HCFcv4u/2S/P5JHw477fD1akWW/mGsZNig4vAOk4135GEWJft9llui8U languages/wasm.min.js
sha384-nwdCbdOL+74sSqBm0yDE9rjqJOm8QqvMFr8uNz2AzmNidbUSRcugcnWOSJEmjsyQ languages/wren.min.js
sha384-eKsHYF6apOh4wN/zUOnYUx5s4086rCTH2ZG0R09R49HLu6/1Kn/6J7Zn6M8++8On languages/x86asm.min.js
sha384-1vxOM0Y2UH442YAG8MrdMqvKlX44D3ay0AWWXt4qBoioszyIQK2MbzVz6OILpqZS languages/xl.min.js
sha384-jgkY4GMNWfQcLLIoP1vg3FWXflDrRhcSXGBW6ONIWC2SOIv5H1Pa57sXs+aomCuZ languages/xml.min.js
sha384-6LS0FTqNBhJALRfpBWUybxuisiSomzK/oLOq6RTNev4R8oJiK2qiX4Liae/Zszmz languages/xquery.min.js
sha384-7YspHobypEnqY8ZOz2b0dwjM3LBhK18ogJfebE+C5BoXfIOO46DczczxY9GHUK0k languages/yaml.min.js
sha384-Pr0oPH6N1VmoBorqbt6dYr4vrTBJcMGglN0EiU/HYRSbqcEUsfF5AM4MF9hbAmEA languages/zephir.min.js
sha384-vx1I5PIkVSkAHbjcpmdlPQm/QmGrcmaqNFopgmjZIaCXdiSOxL84U6Vcccfmf+p8 es/languages/accesslog.min.js
sha384-BeYdlHmYfSfJIZ0BwNBIg4pHN2Nbad4B2nSb9m9bHEM6kacPk5/FlA5ojzShEDLP es/languages/abnf.min.js
sha384-0dpt/84avHAYDCu8+bmVnV2KV6w10X/I5snuSBugTvYgpAAg898GMaCBYW5O3QxX es/languages/ada.min.js
sha384-oVi1X3v1jp9W9YtA/mUcKgEi43Apn1XZwJ7vBkHiGZoK4V0b95pEN10eUjntt3bT es/languages/angelscript.min.js
sha384-st79ydYTL1pcPEFgYDOVomM21FOum4A1cI6cwBhQ8rSAIBTzoFX79LRag4nhMJdH es/languages/1c.min.js
sha384-T5Wi3CdvOgNTDa/TQF3guYZxiCaiLFKzgYxX57ZCt1rVax+G6GP9EOlNdg8spAUs es/languages/apache.min.js
sha384-NRvcTv2Q4lLgjyqBi5tKLK9bHUJh6OJjGeU9dZXroNUOeYSc7ZN2zSr5lfFqcI1N es/languages/actionscript.min.js
sha384-kjROxlwayCv+KEVryJVQjpjchjf7C8tyLrq7hDNnQeBXCp3+X9ZQdw9yXT8wRikI es/languages/applescript.min.js
sha384-Q9japLBVh3V9xw3hAVI+bF2x3uLBMgiF0k7VvXxdXW2tpKQh23X9e74kEWvFpEc+ es/languages/arcade.min.js
sha384-t9QnTos94FrBBrAC7s5wV5c/I48x4nw+3vsiNa9XwkyyQih5RbUUSqt2Gd4uSfkG es/languages/asciidoc.min.js
sha384-pJoNPslHHsgF8oWC+6BTz8z51WuwRvC9FZDIthPkCWH08v+5qV99ZM5rSaiTqEHl es/languages/armasm.min.js
sha384-OwqoHxcABDMucpDcqqtUFQOvCjXVTurrsakiRwMZJGicVyyEYx+zw3vsZmwbSEVo es/languages/aspectj.min.js
sha384-TIMH2N4/8B3XZs9pNBB0s7YlajLJPScymS7grBYIh4eg5HOHRANwlpczewRdiZPD es/languages/autohotkey.min.js
sha384-aRhpIolQegOq4Z1SMG3olwIyLahejSMO3YFexuiuhf5X7w7AK7E78fCnmrsBAN83 es/languages/autoit.min.js
sha384-JUht54FB8uYtGIHcvjUgfQSXni7DprV+V2Bei6BSADqcSdrQLLjj5L7Ih6a8ATJh es/languages/avrasm.min.js
sha384-wxpODyZ7tCYbhfoKuX87Cx+MGIOrV7i3RfCi6GxEs1nhWy9afZQFDq5XzoxKfQFV es/languages/axapta.min.js
sha384-jq1fKOKuZud8ipjshrCh4KB1NERe/+1fdksI04v7A/lXjb3/8Rz0nR9/jwBlTmZY es/languages/bash.min.js
sha384-3EDvjyClCK85gMCNUIHh/H9+K30vwYJjkGmjdUKpG6bGxahP+dSc+HRtXadT8eUD es/languages/basic.min.js
sha384-NAnc3ReMZ27obXeSoozcO7isebTvT46UVK8K3uNQekaf0nmAcMkObTVPVamXqAbg es/languages/bnf.min.js
sha384-UjBtWlHRGRibzEDg1rIFn+Q1OGqWCyMh3RkUN9+hDW27sIB2j8mcGvxmkk8l8LVM es/languages/awk.min.js
sha384-p4Tae887Ng3m5XTZGLjFsoGWFtkRbo89MmMQf70jaN0jwveNEMA7tZRnib4YvhW8 es/languages/brainfuck.min.js
sha384-i9L2RFUCjzYyWbIWd9HedC4OAFnN0QGcMxgy4kJn/sNHygJSPI+4lgD3YKfklGUL es/languages/cal.min.js
sha384-G7WtwjMBNGrPly3ZsbPhfnT0v88hG3Ea9fQhrSWfzj7cNAsXsU7EKMQLyLGj7vTH es/languages/c.min.js
sha384-j1rHZcs0E8MTvcbjBDvgPJ2IWTlTwy90lyS2uAcDbDYEoMjL+OUyqaAPzN8Fe7YN es/languages/capnproto.min.js
sha384-YKGLb4p3/cIXuej2w+oDaefFPxyAVofgsLHznpkId/kfKg29+RKweKKazxWV+3gf es/languages/ceylon.min.js
sha384-7KVG2qZ5ZTMB8I7DeWVIsjFanugFQSryvuNwhwxN1pHBMXEU6LYgpU1ZTvvIs4dD es/languages/clean.min.js
sha384-trRP8CnzW/SSG6kXpTUGPBhUS/tOwRQ992lsIs6/zZ9FueaoJeINicKwxLLVIZKV es/languages/clojure.min.js
sha384-k1X+fuSMpcR5xKe3aKGlyhvbYhLrDZlV/idD0dyH9Ez0QfO+xkhxwQTdTHcsu/ne es/languages/clojure-repl.min.js
sha384-O4LQvDBqImUEPey0HB0Cg3aMyXEf/P4KYm+7uDjqqeU0+ciFh6wEQd3jkRbLD7G7 es/languages/cmake.min.js
sha384-i0H6klsviL0Lu3a5kZ6m/YXMTe2LYJ+I6ZKAsrDhhe4Xl7SvhKEfenOaZJwrBvPa es/languages/coq.min.js
sha384-wm7mUd/Qh4vfF/WEg88/R6WXSI/dIh+I5Ng0sgZ+0Pfg956wqTS1BPzAFBomI3/1 es/languages/cpp.min.js
sha384-u+Rm2j3dViIQNJois1kTeY8JFwQvOscR60DE38HZQ1CaoOKMFJP8bNec7w3U7W/r es/languages/cos.min.js
sha384-FWGvxRI9DV+fT+c7GmrgMrdY0JGqiYLXDhLispUSnhqeHMfnVQXae2ICGhKrsyVz es/languages/crmsh.min.js
sha384-MPtVOX+KeQp5xKpB/zWOsMlc9gRhFxSnI9yci8bfsUMM0iJ7LeiUQn/Y6pWQdXid es/languages/crystal.min.js
sha384-huUb4Ol37G1WrtGV7bn1UArXcJSjD4tQswMGzgpNZYAPxR74MHTqW79z1dXWMvhk es/languages/csharp.min.js
sha384-CMphVcj9wPdvpoNG/a3HS5RAI71czD/WkszUPkiEaiWf0fjZlWmTN/sThO5ZLVt3 es/languages/csp.min.js
sha384-CnT2dpYLO1SfH01DV5VaQHquroPFvb78roOlKOqppKuKp5a8MpL8iOIW19wvZuhU es/languages/d.min.js
sha384-uieyxXCuKHe7O9+gaoPS12wjInN2v7cKQZSKKCcm7mwSDZf5UidK7G4iMHvbZKRe es/languages/dart.min.js
sha384-jXq3YW2qBlEEVJJxkiYe4RHnEk4kkPJu7Wg0Cea5VnZygckFLo0ripESXwXtYD2r es/languages/delphi.min.js
sha384-OIJrZgyhN1E6e29B9M3gFPtkR5JgXpls8h5D2bNzBxQ8Ncwxzfjedu3/0jZEe7S9 es/languages/diff.min.js
sha384-fYhQZafPWO54UUJ6/TWCLZVfptppoJrgZw8syHnvkUeAf7+lAiBkU3x8UQ7rTmvB es/languages/django.min.js
sha384-BZdQkDZ1ac9zSg4XmtL2hLKld98ttTuqoM6CWOjKIrT16NkmNSlAY/1rK48DQjqy es/languages/dns.min.js
sha384-Lr5UvDlMpLCSLEdIc9Jbe7lSHTs1UqFkGAwkacIOzl/Vx34B82LFi+2CeClz1zC/ es/languages/dos.min.js
sha384-DQdaFTXRCa9EQdAkBCHnMAhAomUgizSpJPmuCzoaDJMAshvgxlJF5d7J/ysvc6f/ es/languages/dockerfile.min.js
sha384-oxNLUU6CptOQIukddlIGRE7mhYf+kD1gx1EeiE70rg5netrD1JjSt5BNvRJN3nnL es/languages/dsconfig.min.js
sha384-UuvUHqhG03LaKnRfgywIMsS+lcwqusyfSh9FqB3eN4dm5YDwzRNVSDiWmWRMcz3m es/languages/dts.min.js
sha384-iJ1kGQzDYnnC4EYrWoXiPVYacEjvL6BtUBHateYe/YADV7oUfuIUyEG0jUlkMc/p es/languages/dust.min.js
sha384-jQuTDS10jALGEitp2l3m/2rTJZKkqrZAYnpPu/l1hGq+/7OdVpCLIZFBsqeyLD1k es/languages/ebnf.min.js
sha384-TpeiEEW/MM+zgu0X/O26T/xMSJtm4DAot0uQlZ73FP66AgdlqWO38C/GqpVwo58Y es/languages/elm.min.js
sha384-B1unRJpfius4v1CDo8bNYlnKYL4h9uC7uhSE+fTC4WNSZwBHfHbD47N2bFBurjFF es/languages/elixir.min.js
sha384-4gcpw5sjrwU3Ojec/fllRQd9pKao49meXVCGF9QNr8q9shPlWffEZzHgh44NUpQ9 es/languages/erb.min.js
sha384-8ifOgnb9KNyetOLnSGz4eAGYbTMGxEotFOxIBCWUdWTyUNntY6bOO5ohA189vziS es/languages/erlang-repl.min.js
sha384-R5EduCTSW2/YlL/qKgN18FII8GE9/D4fJnSNpih/gNUzPDoiu8intnPNdulWELkT es/languages/excel.min.js
sha384-vxdmwH7is9bhj+5pQwElHkjcXGSOTsDdv4jaNmNtIcXtACdzVW7BJ7KzhqTJyk8W es/languages/fix.min.js
sha384-qLpZFl0uFVtxGDLGp2q2g5PEgEgt78MggMr+BKUdDtRioP5Uw6RrYO7VzUheZ3ti es/languages/erlang.min.js
sha384-tfyHooH7hV0/4S0EFsbgk/H9JZRSCEOuZSUOLQ6udPcWeHT17xuRlhnDlAPd1VnT es/languages/fortran.min.js
sha384-PJ7Kn/YrJCGUOn0tqrnHoyC+5ovXah3Gj2L9MdsqkmGXKO2VxiJkLYFWH0+7La2V es/languages/flix.min.js
sha384-AxFBr2jCvQAMms4HWRUOeVNg1UdqPDHnPUaboknrDEYjRS0/sjJm+F7jULRPUrOv es/languages/gams.min.js
sha384-zdDMdl9SejIG+Cf9MqovOlWLRwLgYtlH6ePQ6VXOm5pIrdH9y+5erBulXiSVGCVM es/languages/gauss.min.js
sha384-TuBw0tmRHt3almtMx6MlW2O+nHazZN5eWLjBa2JfZ0yUFXXERAQCHXW1Ca6mEk0F es/languages/gcode.min.js
sha384-i3vSOyliC91fFYQu8VD5XHoU0of8Ol45seUxrn0khlEW2VIBcNm3Lu9DrNJzAwWX es/languages/gherkin.min.js
sha384-M7Ef1sJ4SsC67Z22eDV84pAdQRp9VnynFqNtD8xry35mYsHET3AmeJXScLce0iJr es/languages/glsl.min.js
sha384-Q+qImPh7I/MiA1r03Kj40Ikpb6PS1c1seZwQHBynw9bPiuSIxRSTHV5zkRZl4OYc es/languages/golo.min.js
sha384-27jMAcMfx5pzlW2ntRUz1R22f43tLLOnYyDHss8iJBUi23iVzYrxbwQKY+LPU35B es/languages/go.min.js
sha384-sYfgkTuyw8lkHmGdVV7Mw1/C2jkRl3ZKZrSceJsMaqZAG2RNed+pSOfjwMMEIBO7 es/languages/graphql.min.js
sha384-+ilwPpCDudisnRgBeI+XBD7X3bLGw6EZnosF/W6iT/5YyGn0Qwjef5JIZXBkdrnr es/languages/gradle.min.js
sha384-bmK/jYmtEeEMK4iBXwryqTjVz7IHU2bV1CE/vBCO9iDwZupnKChSyiDDi3s/6ZM4 es/languages/gml.min.js
sha384-tgywtm20fLQUu0uIXBr5KDIjyHpfZuyM4zKiakHHNsi9zinDSqPIr/FVjREnEWmY es/languages/groovy.min.js
sha384-8ZC5S/ZP6Df64idPHYBQpnhUfWahrNdU9AYSKhI9hXzYJFfNh5hKZfBNPwDB04Y5 es/languages/haml.min.js
sha384-RNsmJ7WM6P4KLAb4uTYFI84hIr0xBEtHjJaciwHoAsq00NcBk87WXUstUEDJfItU es/languages/handlebars.min.js
sha384-HPIAyvou0DrFHYTekBAbUaQ3UbjeD/WEEJ31SYlgvnEQP54tky7yggE8xQqdfLk0 es/languages/haskell.min.js
sha384-SVFB5K3g6+IUca/JOVZ/3Xb6vzLQ+DEJMC3hxJjKTpbKPLdLhBcaZwL1RY9VaKbI es/languages/haxe.min.js
sha384-sIJo4Fnt+dUnf8gesZEEgKeLE3/+4qdRtxRY9Viz9ww1n4uLtxjSX+wzXBBSFUIY es/languages/hsp.min.js
sha384-GdKwMgTrO0R3SPXhm+DuaN3Qr7WkqYmmTClINAGPuV/BUPE9WUI/WnTEntp522Sl es/languages/http.min.js
sha384-pU9K2/50auLbfR+srPXkvYljzVrrejgxbU+ZgFLo1chaMV/+BWc4VPG6mlNE/c7S es/languages/inform7.min.js
sha384-+nG3MDe+ic9cBObAAKiJtqDoEwEEx49doC34eVj7C3B2Aau6QAX2uden2fob+WPE es/languages/hy.min.js
sha384-LDu6uT3diI3jBUJtdoGFa787cYlVrR+aqFfmW+kW+TImOkpVe+P5LBdDzydIHo9Z es/languages/ini.min.js
sha384-AfNG0QPUG/newzqntC6bt/nCTg5R92zf1gkTVomOylNmjngK7Mmekm3k6Z/yWvuX es/languages/irpf90.min.js
sha384-QxgUyxVKlIKoUKhpv5x52u4Y49438L6rIge8mH2UqBI8S7aCKL7kR+1ulkny45ak es/languages/jboss-cli.min.js
sha384-HM+TR57jHoporR9TWdPZvYUAiU3tp4Nc+A3IoP6AVOrMtcTkDHWzhztA6U130Ap3 es/languages/isbl.min.js
sha384-+DT7AtubDhVDciRc6CgjJJRsCt0L8NC3Dh8n9Tj2tZWU8rWxDIj1ViubmUDn8OCY es/languages/json.min.js
sha384-LTa0cywuUcomDh5o0gVu3KGAtps34A+P7poAvNGIuu1GCw79Q6rFXzyHtbFKK6hl es/languages/julia-repl.min.js
sha384-NqyA97ZywXJCu5WG4NiDyJRAYm5L2aGPPTeGnRSfkEtK8Lch/likdativWWAbLUs es/languages/julia.min.js
sha384-gCVyuWMnFFGQ6thm3jU8iLWbrbIjaaC/iR3u8pCmqgfBF285E47RprBsEz/Y/jlP es/languages/lasso.min.js
sha384-epMw7vS/pc3vOOz4l9Bc4ivmcMrwtBmh3ZRGLa/auLNCxt6IJAJd2HtrsYwa6q2Z es/languages/ldif.min.js
sha384-KyDXPzbgCNyDlWIakKlV0Q1lRw82C2hdumJtLoCQnYfAUwzxxCFzV/AUvjBI+Ab0 es/languages/leaf.min.js
sha384-dDJ50ltJ9VlfSOzSORI0KSNPcRIpLy1J0BmtdQGXBEY+7f57DAx754HVffzmWqC2 es/languages/latex.min.js
sha384-bsXthu50zsdYbXxxCYW0UhAYOakagZBGgi0zpgr/xD6kWCvcteFNRt0VaoJwtPDp es/languages/livecodeserver.min.js
sha384-yR0v1Wvtd+8Y1TA+Tp5e+PEYgonMOQpp92Ox651UghHCxvLkEBAuEBWWCZqm9wGI es/languages/lisp.min.js
sha384-pOSVa4NhOVOXMXeggwYt3iJCS2oCMxcgRPXN10s/V04KOn35iYAATaOrtN+B5KiG es/languages/llvm.min.js
sha384-wvrGnUzHwJzO9dWQFF2DxrFjkSPw5gmc0iOQYmJUzeZ4tqa+15VEFRCH4GI2DNF3 es/languages/lua.min.js
sha384-oH9xSmVdmb6pOXGZFMvf5r7IxH0nNvoScO+WFPIePFkDOYDLLRl9cRwxKD7BbiOQ es/languages/lsl.min.js
sha384-aUXBqKsjOzPD/W+hccF21KKWmWts/CrY/lWGJU+dAcsKtuh1/XtyDnzfLmqy/fV1 es/languages/makefile.min.js
sha384-gU9Mv6FEG6RVi/d3cgP9RYHSdGMkLi/gYQCn7smdbY+m1qDCN2paZoONWRk0tBP4 es/languages/matlab.min.js
sha384-SqGSUq0DMQ0OUQnQnTuVDCJyhANd/MFNj+0PF67S+VXgHpR8A4tPsf/3GoSFRmrx es/languages/markdown.min.js
sha384-b72QkNh8MNwjM/IDSdXmuJ6r37hu6yT4Oo2mDtUYrfEwbKM2UItifB27p57AWsgP es/languages/maxima.min.js
sha384-ttMZ4OF4mQVJDQTomMlTlVLZ0bblHkdalqNJVxD9h0kBRXGYq5iyqo3c9cdRPLkL es/languages/mel.min.js
sha384-A8rbfDB7+vNpdC6fh5/v9nNP7IiJPokmK0dS1rw1Ly75B0PHn7gs/nY2y7SDvr80 es/languages/mercury.min.js
sha384-ycd5/mVifVUTJGtoO7gsl7A/aclmFFPaX1Q8ILpcKKMMlw0n0+7CDMSSJ2XWORrD es/languages/mizar.min.js
sha384-rwr2yhBrQf6ensz5tbrbBa7Toy3hbIrwSLGu63LjSpyACYNBJUzxrUJi2IloSEuM es/languages/monkey.min.js
sha384-UP6D1nnVChEXUcAipJZkjS0JOFnCBLsk8+aUzc9w7HfoC/Mz7Pk36vnL/EqcEVQ+ es/languages/mojolicious.min.js
sha384-0Qx9PV0NMTFk+RqaKeTcoj6XqYesVvfXoezvsEiXyFqeO17iMKoo0vvQ9rRFNRoc es/languages/mipsasm.min.js
sha384-UJ9l8iariH3YOQUdN+8qXsXW2iq+OefoT0mgZVQ4pph/5CBSztTRzdpt4vJ56y2K es/languages/n1ql.min.js
sha384-5995oRqYs6RjByKx4RkQrtptNC60tu7fpQwrvziMk21QomZqyWQ9MvFp/GEoJVGf es/languages/moonscript.min.js
sha384-tpFPEHvbpL3dYF4uFiVNuCUF62TgMzuW65u5mvxnaJYun1sZwxBsKv+EvVgv3yQK es/languages/nginx.min.js
sha384-/SUbXxx72mJbMNPOoNQHdExz/PWgTMuKKUQXZ02bm/qgienw72C81yKJK2hFljfz es/languages/nestedtext.min.js
sha384-trp6hYonRewpJnUz13JwZm+20STvnBrJhyTEN8J59do8eREiQHY2UMDfbBR/L4EO es/languages/nim.min.js
sha384-CtpjUU5MhTp9rxGLye3LiXoBtSBotmKHW9N21wn0Jn945bxMFslJAhlPUb4GUGA8 es/languages/nix.min.js
sha384-16Wkm0YivnMLEbWFVCc7XZ5oI2EKz/82+cGW0lnb4GHN8YD7XBaVcvSGLOeh47lo es/languages/node-repl.min.js
sha384-eSkOKizj8aZA0WJaMmTJA3uqUtTkbZ7PXAADzDquzxCLyb1BMNgF9cn7/FIu0wFR es/languages/objectivec.min.js
sha384-sm+3C97241bDJHul+IONVgYeQJcwtSN2wo1kUa1K0mNWY6MPMJnrx1giFrUr2MNY es/languages/ocaml.min.js
sha384-r3EeYX3CbwY0FnOv7LgNiecjqOoDdFWvHW4EPlPtZpNRd2jbxdiQKH4yzDmiKNsG es/languages/openscad.min.js
sha384-ExJ3pccZj2F/NU0rp4Y2OvfEgWFmLtjnUZ9JRGTrwAT59xVowpt5JZBoesVs0a50 es/languages/oxygene.min.js
sha384-Or8qgh2b9O8VXzQ3a0tiaxFWFv/3TNtMDBjuXJJBZ6eQQgcefh/+4/BjwuioV7zy es/languages/parser3.min.js
sha384-HN/8Qf4NvPiUakM8jkY+R/WaNf5uDeDa80fK/VDjoi9huy+nhUzrTcWCFDjNUrGI es/languages/perl.min.js
sha384-zOSNwldQX3ufsVxttzdDmDOl9B8zrMIbr7l0ixGfdle/Ree4OxUJK7kQjCDqT45z es/languages/pgsql.min.js
sha384-OgdYEzbHjXEECvHgZe4NRcmfBJjT3ZPVSGLUFo6xZe4DmBtLi9atHdaBdBnsPi+O es/languages/pf.min.js
sha384-dGJwrvnfo/JaY0lSDtJuhB6XXPmL9hSqOnIYndlN0eE1PoA9FFxd+VtaSfcacNDA es/languages/php-template.min.js
sha384-6Znh5S5q9F0hGNMtdaihFL3RJeMqL1a/UecjgzaBrhoxe4sjd5aiEOgokt7blhSW es/languages/php.min.js
sha384-t0XeJ1PaBwN1QukCYAP/EPBK7SXIMRxw2R2nm/P15RGqdxjFIFmW9+7i/Ilm9Xqa es/languages/processing.min.js
sha384-XXx7wj9KPm08AyGoGzzFKZP2S7S+S5MbKMPnQcWUyhJ3EjHvLuctK/O1ioJnG2ef es/languages/plaintext.min.js
sha384-pBczxETXXX/Ne92jpBviT47DPiv27FLNtvs0aMZH3W7rYXdwgf1gWg3pH4NmNC2V es/languages/powershell.min.js
sha384-ugBQHWNIckVgJHtf5sAjmzt1u/UqtQYmF0di9BbI9W3K8EASkdIHjrkFhv9XiLEl es/languages/pony.min.js
sha384-MGwRmP+fUOD51bd1a03edAvcc1Aaq4Ddq8WnyV/cMWm7e2QIRDSkSOGOUAt5u06j es/languages/prolog.min.js
sha384-KN6bgPY9F1QInwObKCM1rAyr3deHf/g5fYoc9VYe1dY4luss3KrzYoDcdI725dDy es/languages/profile.min.js
sha384-93TO3rpvKyXqikneOt71cSHkxWQWPKaa6n00Me0j5Fzvjb7j+EjURqbZHyqx8m8T es/languages/protobuf.min.js
sha384-Drd5kfuBp36sTS9lcJ+ai2UhYgBwhpYsRDl2T3useOowrqih3gTCBLSQI3M5lQQm es/languages/properties.min.js
sha384-JrWCxwBg3eui2fINXeS94URdHjdsdop1UkpW9t3kFMya0i4jXubaG9QZKghbzQ4x es/languages/purebasic.min.js
sha384-AD3TodU3dq//GRWPVK0bQi/Px1bVpPbCnAmqJ5eh95PeV2hmitDzKvL1PI39Q0n/ es/languages/puppet.min.js
sha384-sW9zIYfNc+qN/lya5oAmIdFlVbXjPLcdYkPSdRMpxO/xI2jbDW/Q4Etvx02DFOmf es/languages/python-repl.min.js
sha384-cqUXYrWVhLbvuAygiWuVOYUcb3YMXXkFnUYSOP1CpfFYnZHH1d5NmMnt7CL6t9I6 es/languages/q.min.js
sha384-rpBTxqcMTDMGwLGTQeWiHl2lU2e8jPaiRl9k8L0T1VAQRlzj7vHGDk7zO+dfQ4y8 es/languages/python.min.js
sha384-UIbzqTGmyjC7ne73j3bNaZjwL+epaAcCbVrBkDbPn7ao4HG6x5u6IsC7B/bRA5Cw es/languages/qml.min.js
sha384-T5hv8+GOm2ni1B0cuZtt+8T0cIMH1CTycc9OQQH0GLcIQhmeBosyhcmcIv4r/1ag es/languages/r.min.js
sha384-5LQ8XZbaR3kmVh0wCOpHq1l5hX9jiEwdtHnM6BsayLhjr8UF1MaJBqcaDeSynhPv es/languages/roboconf.min.js
sha384-l9wgOjfWCEirFouYdCSgnidjAsYEYXI7Sfl4PMP3F1SmZrvGDDuaS7L/LahMxqZB es/languages/reasonml.min.js
sha384-v1pmgA1WUGCX5tWByVH1SCIAkvNz6hnl0bVLssfLEC1jds/85mpYLpvu99xxmeAj es/languages/rib.min.js
sha384-Bn8/cw9AY/FdwiTs68/oQSiKdIeN11iKHIF2dvLLvhxFTERT0bJxQ0cvw/4FUY5B es/languages/routeros.min.js
sha384-D/4t4rRgGugSG1pilViiFGJ/yatWQdXHmulH5MMYSGbGyjMkkpc7kmkTuq0ObGfH es/languages/rsl.min.js
sha384-O4a+vELXT191NhKLE3TR5WQwDmQZ2izAhb2zETRxcPSXr6CJruqJ4a+GJpDlaqCF es/languages/ruby.min.js
sha384-aO+3EnTqFaW1TRPKbxz6zCJ7qyW/qoXfmBfchbqx1Vby3kEykZNMoMOox/qY6P7O es/languages/ruleslanguage.min.js
sha384-oFPOq+7PHmif78WVa8FQSBHx40SkpM3OYcl5VT4GLz0751RlRpPzp2jfC9XFJOi+ es/languages/sas.min.js
sha384-5KqbJwOue0VrSIbMNJrkRcBOzNPBxlGPbrTdH/biCYczeCVi0Ethcg+fr1RXfH8i es/languages/scala.min.js
sha384-R+AmncR/y/QwiCROx1hLg1nFTQWR/07Yj4jBb3dhTFzINFaRfhpkB8G5+J4aIkrq es/languages/rust.min.js
sha384-lEVNmnotVzJyk7ga6toEDtsBpgn9WoqwbsxeCyDEsmdGqyDBNeWI1kFiqtJ8lJF8 es/languages/scheme.min.js
sha384-Gd69hZP/y69jKMIzcFdL+d2yjk7GiFD2i5DXFPnW/ThV/DDzklRMCm2BKhEV4zBi es/languages/scilab.min.js
sha384-RKUoelG22/D7BV/bNpoGLNzdTgWRf/ACQX7y4BGyIzK6E+xUoXtm68WNQW2tSW8X es/languages/shell.min.js
sha384-1hOjuVtr+BywO/YBxY8v83dP5hXcva8MF+HtG8/3fOAtINMOqEEfiDNWbTBy65HJ es/languages/sml.min.js
sha384-u0+HbQcahC6m0dedTQHD/+4ltin4W9rjZQxNGBEW/6/ORehGOvu1LXEkhWWrLb3W es/languages/smali.min.js
sha384-v/u/OLs9OhRDvTsH6gl5i3C0/an5nTMlCRkkLw28NuaXWI1BuiNlCP24cXYdNOH8 es/languages/smalltalk.min.js
sha384-8G3qMPeOeXVKZ0wGzMQHgMVQWixLw3EXFAcU+IFNLRe0WoZB5St3L3ZLTK62Nzns es/languages/sql.min.js
sha384-Oo1bS36hboiUfGd7+uR/KDMUQb5cU6NaRGLTkoJDO8/yhKAgAcBhp+l2+6PXmHcG es/languages/sqf.min.js
sha384-8BaNxuWwajbXmJkNYTbaPvAqor6+1+4k41vhrbt28LWNtCWNOxYeOA/FzToB8zuj es/languages/stata.min.js
sha384-KR6+6nkAg/MOQBesihYKeMGwAwULn9mJbO9/dogD9cqV1EFz+zpUK0dELhLhWnLC es/languages/stan.min.js
sha384-dw9W6SpHrOZiHIct83snAFa8wooEfYm3po9ObJQjD7xZOqa8VA5aNvQN3ck7Jc97 es/languages/step21.min.js
sha384-3Y6/blsro4snWHovFrMSw2x4A0R3jveIhu3My2Bb74xsDlas3Fa1FhFY55ti0ZNQ es/languages/subunit.min.js
sha384-P8/Fqc3P8z04WoUH54E5f8QNfkGEGUG5ttSAArezchkW1ppTXJT4tJqfc1Qbph0a es/languages/taggerscript.min.js
sha384-04sj7r6yrNuUHoJKU+hsM7qLMIpfko/AyyA7awDn3DNA8PVuP6G8cXbPeGbYt6/b es/languages/tap.min.js
sha384-fdDOAbgCCSuCQZYLNBFr1Iq9P8nOC+SR8iBVSCfMbSAHuZjNmqZjtW7etWgoOHZN es/languages/tcl.min.js
sha384-0cID1UeZU+LTU4+4kG7dHQLZybx/s882jt5nCRhuGxt/jcrsuyT+yud82WOKVOgp es/languages/thrift.min.js
sha384-dKtXt9yuyKTD7gJVf9rOJM+Kvt4yjn5r6p6xHHUIKySEQCv9XFtSLbUXrOkL7ONA es/languages/tp.min.js
sha384-0Thic7W/L8oBMGLVGw9CIPWYyk9e8fylWAilp+lwcBsL9QJ0uojQysaS7LBJBpRH es/languages/vala.min.js
sha384-Bll0N6Bv3PBNceR+4FI/xj/vKUrRYkGyy0oHArQg7yjocZ4goBU/MlbynwkpxynM es/languages/twig.min.js
sha384-LdqDRVWxc7cPE2ISZcBIeHoQAyfnXmEArz0layp8fOjceievL8eX9cV+HgneOkQ4 es/languages/vbnet.min.js
sha384-7rzHs2v9fu5E3xKT9BbxDXT4GNQvENpY8IWOVCKnwNvogY6AGwl92Ij/rt8Omiyq es/languages/vbscript-html.min.js
sha384-+GEu9SSqFFm1Oorzq3xLm59JzaVcoELsmjRU9e2/5+qKtgBvEqt98SfCW5RQ2vQ7 es/languages/vbscript.min.js
sha384-3BuLJ1KBGCbYiSNyOuKZqvmJTYVibt7rw/vg5wX6QnRIJEV0s2fQt7nYSvueb6FH es/languages/vhdl.min.js
sha384-6V6Uo2UWsH6I18erq91BIHdtSPcnACqWo809ivdHuVgXpqWAL/C82DCCnPijsTtD es/languages/verilog.min.js
sha384-8kEj1LOKOKcz1ifO10aWlws8zNCnrxWFhHS5zWENNU+dOB/z0X/X31StZHBTQ64Z es/languages/vim.min.js
sha384-FlAOV+6BVjUyajzW4twmqUvHxDYallf5jw5FewMZmsBHBuiJ/CJj1NZhNT1fHuzT es/languages/wasm.min.js
sha384-EAsq5ujCzyOLI/Q/1sdpsLcFIcb9GNiIxIQfxnkEyRNzswn9/MI+wJyBvHEZdbvT es/languages/wren.min.js
sha384-POvW54o3Z6xZV0L2TM4VgkaKxfrnDTCf3nfOeA+Szz73C2dm7jL8CRGlBtM0FHMr es/languages/x86asm.min.js
sha384-k35VPb72MHlqV2SHnJK/VO7pagipEvXPoGyJ2Cohs/MU2aN9Hkzjzw0fpwbOUXgq es/languages/xl.min.js
sha384-yFd3InBtG6WtAVgIl6iIdFKis8HmMC7GbbronB4lHJq3OLef3U8K9puak6MuVZqx es/languages/xml.min.js
sha384-ykAvQddsMz2kd8fP4oPW6ZvTLCXUu7Z5PLw13CLPULaTYic99di82Qpz19z6pP8Q es/languages/xquery.min.js
sha384-MwRFV0kWo2NtUY8PZBaqrbQuA/FB7ouS40j2CKvHWA2W+ODg9LKjrxXWIlRekoVS es/languages/yaml.min.js
sha384-iEOx9ajh9HkrFMP+goJQl2Dgl9iV2wpFywezRXb9E0sm5PIHETchiP9xFLgcYekX es/languages/zephir.min.js
sha384-n6hF4L8nW2gwnRS9wgPCaqkEH5PDOpgY3LnNStI4UlzxymSaagYyvGqXsr6E074H languages/arduino.min.js
sha384-v78fkUxICHap3erbZ72DyAoBlRVfNKVpjsG2fwRS+ELQNoSWDe0ppa1vTktch8gy languages/fsharp.min.js
sha384-uDBWF8my1Ej38r/wxEQnnSxDowPvDtRmk02jvdvo/SbRnc+9O+0aSlmnZvIMT8vD languages/nsis.min.js
sha384-LAzSkWYZ3FWRTDwn+LHzyO1jGUFT5FYSAABzOTE4lsm429sEXm0+U7A5i73hhMSz languages/coffeescript.min.js
sha384-44q2s9jxk8W5N9gAB0yn7UYLi9E2oVw8eHyaTZLkDS3WuZM/AttkAiVj6JoZuGS4 languages/javascript.min.js
sha384-6urPKs3QiIWOKSqCbj5VXHsUgaLczdBHaUjm2olYRsv0Pyy5Y17Ejr8Lwq8v1pSq languages/livescript.min.js
sha384-EO/jSdjpPTVnKfaKudbmOTZBYv471ACLOeLdy0zhCEJrQAtjgUSCbn/Cc+9QWjIx languages/css.min.js
sha384-a9eWSfPP5RET4jGydDkNQvLLFuVZFOxYOsukKgE63rboxeChYr7XU5lh4A3yhLvs languages/less.min.js
sha384-P21M4eXuFNieGV4NM5GbASGWGs057r9VCRBls/B/Tkn2+rYO+U9yhq6ad4xR2Ph+ languages/scss.min.js
sha384-MeA3Pq3j4quMdULL4vWWO96ixFxQwcYLr32m2j55WoaO/zGGrJF9H45Gd19Bv4OK languages/stylus.min.js
sha384-uUg+ux8epe42611RSvEkMX2gvEkMdw+l6xG5Z/aQriABp38RLyF9MjDZtlTlMuQY languages/java.min.js
sha384-vfngjS9mwSs6HkzR9wU3mDDip7sa8TLKAxsuQ9+ncUHU25slHzHOdx/0FWxvbg4I languages/kotlin.min.js
sha384-Y0PVkUaoJo8lSq1gzhxTxCx58Dd5w7lB/0RKYuIK7hmiVyndY9vplBBava32RP+m languages/mathematica.min.js
sha384-jO2AD2ITIy8nYCrwS1wOk1JOkNsyYJZ168rudfUdsPBJct4i8KR9rWtsA9YbozN1 es/languages/arduino.min.js
sha384-B1ZRrYmfEhHwYo+zuVPZpMfzy6cQt14Vvsbz0ZgQaJfZotA1AjdYP0C+zvN39Czh es/languages/fsharp.min.js
sha384-I6QuqZHBmq0/uzdbM+QDZ73Eh804X1HzoFdvMYxSLla2z6hBzg6hpa4nGKvJoEfX es/languages/nsis.min.js
sha384-hs3Egk+9CzC2kK7ZZLBVUrgiyFcI/OzhECFhJzoD2Qm2ipxDbrwCTQAh/RTkuo6c es/languages/coffeescript.min.js
sha384-eGsBtetyKPDKaLiTnxTzhSzTFM6A/yjHBQIj4rAMVaLPKW5tJb8U6XLr/AikCPd+ es/languages/javascript.min.js
sha384-zTxPANr6JZxOl1dbhuPLVWeBBmKtzXx5ojS2jsuBNIhT0fsDuSjFkJq3T1ioZVdy es/languages/livescript.min.js
sha384-lt+tmYaOzXx9MmAecQ5yQIclKAWSYC5cYi6wxnfEX2wR5WmdhW1+ahpdu+/Ka5JN es/languages/css.min.js
sha384-hcTAUmAPcLa6fhkSwRgJpVDvoVF9afjydQiXyQYPDyFKTn4bATMWaseBh36/7wU7 es/languages/less.min.js
sha384-kgS8WQ5/fDJ7xH5h1sT4IbnIn7mJZ7RExzG5Td2d8GuVg7DhFoiijDcC8HTxJs2B es/languages/scss.min.js
sha384-N0SnsP7J6c42VNecxNu2Gf/wu1z5fqSUtn1jScgNVc2mwN2f2iXJfeXCh+19JhUb es/languages/stylus.min.js
sha384-coaxfgI2lKuDqSxfMlfyPq5WM0THaLGyATZHzaFMrWdIbPcLdduuItTe6AmT/m33 es/languages/java.min.js
sha384-mfmdbPhLobPr5OJzSXlWHDQDymRYyzedurWjJBvKVhlQGE+Jz/pN3D9lPEBIkDK2 es/languages/kotlin.min.js
sha384-ZOli511kLeGbbzCcuVX7mH7u8A+Vv5xoT3HaBPcSW8DwrsJkjCA7Ln+Wluj9Xm17 es/languages/mathematica.min.js
sha384-CYmrQ9dmDVxuVoM185jHQsjhiLlG/kQfabzRdOYsfUV2AQvpjQNrd2zVCpCC7N5j languages/swift.min.js
sha384-ORwtVEfrCZ0gzGacgmfv1wOtxcPIaVfHKwq8dKQjObRwx3qpKjsSg1ldTu1PEgXd languages/typescript.min.js
sha384-lJj3aAxzUpdk8StXN5j3OP20/Loadv+t8jYdMBYVqCaxtLHTpBUalFDsTPkC9Fov es/languages/swift.min.js
sha384-HHgazax8ozQ9RDWlJQyaFzaTk/OgTFRVHH+lcaYInkE8wVu5fnpkqkB3KUdzKcvE es/languages/typescript.min.js
```
