<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.

/**
 * Anobody can login with any password.
 *
 * @package auth_jmoodle
 * @author Agus Tri Haryono
 * @license http://www.gnu.org/copyleft/gpl.html GNU Public License
 */

defined('MOODLE_INTERNAL') || die();

require_once($CFG->libdir.'/authlib.php');

/**
 * Plugin for no authentication.
 */
class auth_plugin_jmoodle extends auth_plugin_base {

    /**
     * @var bool Web service token created for special user for interactions with Joomla.
     */
    private $joomla_user_token = false;

    /**
     * Constructor.
     */
    public function __construct() {
        $this->authtype = 'jmoodle';
        $this->config = get_config('auth_jmoodle');
        $this->_getJoomlaUserWsToken();
    }

    /**
     * Return a web service token created for special user for interactions with Joomla.
     * Please, specify a special user id and web service id in auth/jmoodle settings
     *
     * @return string
     */
    public function getJoomlaUserWsToken(): string
    {
        return $this->joomla_user_token;
    }

    private function _getJoomlaUserWsToken()
    {
        $joomla_user_id = get_config('auth_jmoodle', 'user_id');
        $moodle_external_service_id = get_config('auth_jmoodle', 'externalserviceid');
        if(!empty($joomla_user_id) && !empty($moodle_external_service_id)){

        }
        global $DB;
        // Check if a token has already been created for this user and this service.
        $conditions = [
            'userid' => $joomla_user_id,
            'externalserviceid' => $moodle_external_service_id,
        ];
        $token = $DB->get_record('external_tokens', $conditions, 'token');
        if(!empty($token->token)){
            $this->setJoomlaUserToken($token->token);
        }
    }

    private function setJoomlaUserToken(string $joomla_user_token)
    {
        $this->joomla_user_token = $joomla_user_token;

        return $this;
    }

    /**
     * Returns true if the username and password work or don't exist and false
     * if the user exists and the password is wrong.
     *
     * @param string $username The username
     * @param string $password The password
     * @return bool Authentication success or failure.
     */
//    function user_login ($username, $password) {
//
//        $params_MHS = array(
//                        'userid' => $username,
//                        'password' => $password,
//                        'usertype' => 'MHS'
//                        );
//        $result_MHS = $this->request($this->config->serverurl,$params_MHS);
//
//        $params_DSN = array(
//                        'userid' => $username,
//                        'password' => $password,
//                        'usertype' => 'DSN'
//                        );
//        $result_DSN = $this->request($this->config->serverurl,$params_DSN);
//
//        if(isset($result_MHS->sessionId) or isset($result_DSN->sessionId))
//            return true;
//        else
//            return false;
//    }

    public function request($serverurl, $params = []) {

        $data = json_encode($params);

        $headers = [];
        array_push($headers,"Content-Type: application/json");
        array_push($headers, "Content-Length: ".strlen($data));
//        array_push($headers, "\r\n");

        $ch = curl_init();
        curl_setopt( $ch, CURLOPT_URL, $serverurl); // URL to post to.
        curl_setopt( $ch, CURLOPT_RETURNTRANSFER, true ); // return into a variable.
        curl_setopt( $ch, CURLOPT_HTTPHEADER, $headers ); // custom headers, see above.
        curl_setopt( $ch, CURLOPT_POSTFIELDS, $data );
        curl_setopt( $ch, CURLOPT_CUSTOMREQUEST, 'POST' ); // This POST is special, and uses its specified Content-type.
        $useragent =         \core_useragent::get_moodlebot_useragent();
        curl_setopt( $ch, CURLOPT_USERAGENT, $useragent ); // User agent

        // Use proxy if one is configured.
        if (!empty($CFG->proxyhost)) {
            if (empty($CFG->proxyport)) {
                curl_setopt($ch, CURLOPT_PROXY, $CFG->proxyhost);
            } else {

                curl_setopt($ch, CURLOPT_PROXY, $CFG->proxyhost.':'.$CFG->proxyport);
            }
            curl_setopt($ch, CURLOPT_HTTPPROXYTUNNEL, false);
        }

        $response = curl_exec( $ch ); // Run!
        curl_close($ch);

        $response = trim ($response);
        $data = json_decode($response, true);
        return $data;
    }

    /**
    * This plugin is intended only to authenticate users.
    * User synchronization must be done by external service,
    * using Moodle's webservices.
    *
    * @param progress_trace $trace
    * @param bool $doupdates  Optional: set to true to force an update of existing accounts
    * @return int 0 means success, 1 means failure
    */
    public function sync_users(progress_trace $trace, $doupdates = false) {
        return true;
    }

    public function get_userinfo($username) {
        return array();
    }

    public function is_synchronised_with_external() {
        return false;
    }

    function prevent_local_passwords() {
        return true;
    }

    /**
     * Returns true if this authentication plugin is 'internal'.
     *
     * @return bool
     */
    function is_internal() {
        return false;
    }

    /**
     * Returns true if this authentication plugin can change the user's
     * password.
     *
     * @return bool
     */
    function can_change_password() {
        return false;
    }

    /**
     * Returns the URL for changing the user's pw, or empty if the default can
     * be used.
     *
     * @return moodle_url
     */
    function change_password_url() {
        if (isset($this->config->changepasswordurl) && !empty($this->config->changepasswordurl)) {
            return new moodle_url($this->config->changepasswordurl);
        } else {
            return null;
        }
    }

    /**
     * Returns true if plugin allows resetting of internal password.
     *
     * @return bool
     */
    function can_reset_password() {
        return false;
    }

    /**
     * Returns true if plugin can be manually set.
     *
     * @return bool
     */
    function can_be_manually_set() {
        return true;
    }

}


