<?php

/**
 * @package     System - WT Telegram bot - Content
 * @version     1.0.1
 * @Author      Sergey Tolkachyov, https://web-tolk.ru
 * @copyright   Copyright (C) 2024 Sergey Tolkachyov
 * @license     GNU/GPL 3
 * @since       1.0.0
 */

namespace Joomla\Plugin\System\Wttelegrambotcontent\Extension;

// No direct access
defined('_JEXEC') or die;

use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\HTML\Registry;
use Joomla\CMS\Plugin\CMSPlugin;
use Joomla\CMS\Toolbar\Button\BasicButton;
use Joomla\Event\Event;
use Joomla\Event\SubscriberInterface;
use Joomla\Component\Content\Site\Helper\RouteHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Factory;


final class Wttelegrambotcontent extends CMSPlugin implements SubscriberInterface
{
	protected $autoloadLanguage = true;

	/**
	 *
	 * @return array
	 *
	 * @throws \Exception
	 * @since 4.1.0
	 *
	 */
	public static function getSubscribedEvents(): array
	{
		return [
			'onContentAfterSave'         => 'onContentAfterSave',
			'onContentChangeState'       => 'onContentChangeState',
			'onAfterDispatch'            => 'onAfterDispatch',
			'onAjaxWttelegrambotcontent' => 'onAjaxWttelegrambotcontent'
		];

	}


	public function prepareArticle($article): void
	{
		$message  = '<b>' . htmlspecialchars($article->title, ENT_COMPAT, 'UTF-8') . '</b>' . PHP_EOL;
		$message  .= (!empty($article->introtext) ? $article->introtext : $article->fulltext);
		$message  = HTMLHelper::_('content.prepare', $message, '', 'com_content.article');
		$linkMode = $this->getApplication()->get('force_ssl', 0) >= 1 ? Route::TLS_FORCE : Route::TLS_IGNORE;

		$this->getApplication()->getLanguage()->load('com_content', JPATH_SITE, $article->language, true);

		if (isset(Factory::$language))
		{
			Factory::getLanguage()->load('com_content', JPATH_SITE, $article->language, true);
		}

		$link = HTMLHelper::link(
			Route::link(
				'site',
				RouteHelper::getArticleRoute($article->id, $article->catid, $article->language),
				true,
				$linkMode,
				true
			), Text::_('COM_CONTENT_READ_MORE'));


		$images = [];

		$articleImages = json_decode($article->images);

		if (!empty($articleImages->image_intro) || !empty($articleImages->image_fulltext))
		{
			if (!empty($articleImages->image_intro))
			{
				$image = $articleImages->image_intro;
			}
			elseif (!empty($articleImages->image_fulltext))
			{
				$image = $articleImages->image_fulltext;
			}

			$image  = HTMLHelper::cleanImageURL($image);
			$images = [$image->url];

		}

		$message_params = [
			'context' => 'com_content.article',
			'item_id' => $article->id
		];

		/** @var  object $result Telegram response */
		$result = $this->sendMessage($message, $images, $link, $message_params);

	}

	/**
	 * @param   string  $message         Text message with HTML markup
	 * @param   array   $images          Array of paths to images like '/images/image.jpg'
	 * @param   array   $link            Link HTML code: <a href='https://site.com'>Link text</a>
	 * @param   array   $message_params  Array of params. For upcoming features
	 *
	 * @return mixed
	 *
	 * @since 1.0.0
	 */
	private function sendMessage(string $message = '', array $images = [], string $link = '', array $message_params = [])
	{

		$event = \Joomla\CMS\Event\AbstractEvent::create('onWttelegrambotSendMessage',
			[
				'subject' => $this,
				'message' => $message,
				'images'  => $images,
				'link'    => $link,
				'params'  => $message_params
			]
		);
		$this->getApplication()->getDispatcher()->dispatch($event->getName(), $event);

		return $event->getArgument('result', []);
	}


	/**
	 * After save content logging method
	 * This method adds a record to #__action_logs contains (message, date, context, user)
	 * Method is called right after the content is saved
	 *
	 * @param   string   $context  The context of the content passed to the plugin
	 * @param   object   $article  A \Joomla\CMS\Table\Table object
	 * @param   boolean  $isNew    If the content is just about to be created
	 *
	 * @return  void
	 *
	 * @since   3.9.0
	 */
	public function onContentAfterSave(Event $event): void
	{
//		[$context, $article, $isNew] = array_values($event->getArguments());
//
//		if ($context !== 'com_content.article')
//		{
//			return;
//		}
//
//		if ($isNew)
//		{
//
//		}
//		else
//		{
//
//		}
//
//		$this->prepareArticle($article);
	}

	public function onContentChangeState(Event $event): void
	{

		/*	[$context, $ids, $value] = array_values($event->getArguments());


			$message = '<strong>Console - Save old articles aliases</strong>
	Плагин обновляет алиасы материалов Joomla, добавляя к ним id материала, согласно старым правилам роутинга Joomla. Это позволит сохранить старые URL материалов и при этом перевести сайт Joomla на новый роутер.
	 <a href="https://web-tolk.ru">https://web-tolk.ru</a>';
			$images  = [
				'/images/swjprojects/projects/1/ru-RU/icon.jpg',
			];

			$result = $this->sendMessage($message, $images, $link);*/

	}

	/**
	 * Add a button to Joomla Toolbar for sending to Telegram via ajax
	 *
	 * @since 1.0.0
	 */
	public function onAfterDispatch(): void
	{
		if (!$this->getApplication()->isClient('administrator'))
		{
			return;
		}

		if ($this->getApplication()->getInput()->get('option') !== 'com_content')
		{
			return;
		}
		if ((new \Joomla\CMS\Version())->isCompatible('5.0.0'))
		{
			// Joomla 5
			$toolbar = $this->getApplication()->getDocument()->getToolbar('toolbar');
			$icon = 'fa-brands fa-telegram';
		} else {
			// Joomla 4
			$toolbar = \Joomla\CMS\Toolbar\Toolbar::getInstance();
			$icon = 'fab fa-telegram';
		}

		
		
		$lang = $this->getApplication()->getLanguage('site');
		$tag  = $lang->getTag();
		$this->getApplication()->getLanguage()->load('plg_system_wttelegrambotcontent', JPATH_ADMINISTRATOR, $tag, true);

		if (isset(Factory::$language))
		{
			Factory::getLanguage()->load('plg_system_wttelegrambotcontent', JPATH_ADMINISTRATOR, $tag, true);
		}
		
		
		
		$button  = (new BasicButton('send-to-telegram'))
			->text(Text::_('PLG_WTTELEGRAMBOTCONTENT_BUTTON_LABEL'))
			->icon($icon)
			->onclick("window.wtTelegrambotContent()");
		$toolbar->appendButton($button);

		/** @var Joomla\CMS\WebAsset\WebAssetManager $wa */
		$wa = $this->getApplication()->getDocument()->getWebAssetManager();
		$wa->registerAndUseScript('wttelegrambot.content.ajax.send', 'plg_system_wttelegrambotcontent/ajaxsend.js');
	}

	/**
	 * Main ajax job. Send to Telegram via ajax.
	 * Send the array $article_ids here
	 *
	 * @param   Event  $event
	 *
	 *
	 * @since 1.0.0
	 */
	public function onAjaxWttelegrambotcontent(Event $event):void
	{
		if (!$this->getApplication()->isClient('administrator'))
		{
			return;
		}

		$data          = $this->getApplication()->getInput()->json->getArray();
		$article_ids   = $data['article_ids'];
		if(!count($article_ids))
		{
			$event->setArgument('result', false);
			return;
		}

		$sent_articles = [];
		foreach ($article_ids as $article_id)
		{
			$model = $this->getApplication()->bootComponent('com_content')
				->getMVCFactory()
				->createModel('Article', 'Administrator', ['ignore_request' => true]);
			$model->setState('params', (new Registry()));

			$article         = $model->getItem($article_id);
			$article->images = json_encode($article->images);

			if ($article->state)
			{
				$this->prepareArticle($article);
				$sent_articles[] = $article->id;
			}
			else
			{
				$this->getApplication()->enqueueMessage('Article ' . $article->title . ' is not published. It was not sent to Telegram', 'danger');
			}
		}

		$event->setArgument('result', ['sent_articles' => $sent_articles]);

	}
}