## Subresource Integrity

If you are loading Highlight.js via CDN you may wish to use [Subresource Integrity](https://developer.mozilla.org/en-US/docs/Web/Security/Subresource_Integrity) to guarantee that you are using a legimitate build of the library.

To do this you simply need to add the `integrity` attribute for each JavaScript file you download via CDN. These digests are used by the browser to confirm the files downloaded have not been modified.

```html
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.10.0/highlight.min.js"
  integrity="sha384-pGqTJHE/m20W4oDrfxTVzOutpMhjK3uP/0lReY0Jq/KInpuJSXUnk4WAYbciCLqT"></script>
<!-- including any other grammars you might need to load -->
<script
  src="//cdnjs.cloudflare.com/ajax/libs/highlight.js/11.10.0/languages/go.min.js"
  integrity="sha384-Mtb4EH3R9NMDME1sPQALOYR8KGqwrXAtmc6XGxDd0XaXB23irPKsuET0JjZt5utI"></script>
```

The full list of digests for every file can be found below.

### Digests

```
sha384-3jXXSCoHN1lT4/D808fSn2SrsDIHD6O22/ptb0bHfRk+OWGarp6aTeCxKn8tvGzD /es/languages/1c.js
sha384-OSLa36uZ8MoDqW/oUAJd5PV1sHIr0nDREiPwSlmVEa82KJDAVc6zTpFt+/zSNJf0 /es/languages/1c.min.js
sha384-KAt+RcjEvaKv42/rvN7MC0d6aZu1mZ6UcWGSybxSWerUl5btuVBtT3Bw7CMinyAl /es/languages/abnf.js
sha384-wKaGu1ROMZaxQzYoWdEpbgPUpNoyGYkIEIZdhHnNN5NI19e02sXHIifCPwQ/ecdj /es/languages/abnf.min.js
sha384-HXa6xmJHMEfHIi/hDPtTbigJQtXSLC1v3Leof82ZGiiIQHcgvyioqE8wgdLm/Mgi /es/languages/accesslog.js
sha384-QqfeezSFk3jFhiaBPAGWdCdQGBdBckwrd9PXxjsb1OObrgikKVL+tcnJYLsSReEH /es/languages/accesslog.min.js
sha384-7MElxj9C++xedcf+m+rMAU7OnoQdOCGGuzuEgDI0dQtcFj2D6tAJM3tHp3fxm9OM /es/languages/actionscript.js
sha384-xVGCBpZS3OpdCcbBo14EIYO8M5OMnVTQF4yKcGp0grCGx6EQkkM7b+ufTu6OCDpf /es/languages/actionscript.min.js
sha384-4O4F7O+Nmg/xBAXYrtACgp9Gf+6tGkxaDdZN8rLxj6oNysaWRbBF67lR1mA8wCuD /es/languages/ada.js
sha384-8EECOILmBYlAkigZ+z/CNnC/mSVvoGBCdsfhUIp9Eh2nw228k0J27PUwfzzrXVwJ /es/languages/ada.min.js
sha384-qM1fV5U/9GJR9QjnTlpp89zRgnYrXLgwKeLKwoQ2PUNTMhxLQMM5eXvfhFFVZnA2 /es/languages/angelscript.js
sha384-gybzJRGZrWV/j50oGMw124LpN/Tc/C+33TDgAq/xuPRDKEgJufzztHa5DdmMks2u /es/languages/angelscript.min.js
sha384-h+WRl4BTTVy31uw0ZuOybtJsEIlOq2Q4LSWUjK8gKrhdXEqYHP7Qu+MmBIZIqZtO /es/languages/apache.js
sha384-xuOg8cLlaKYVwu8KcJCX2nbs9mrVa7Ngz0c9NQiwViwl408M5TX3w1FC22RHhUuS /es/languages/apache.min.js
sha384-zSqDrr2HUSwHCgmRiMQUtbc4a9qpKFct6B3w54ZZNGTsBuHy0PUufQmLF4rLwY6X /es/languages/applescript.js
sha384-2kZ9vXS4PfcoGlwJ5gNbv5QnUY15tcWNHxJePONFXmRi7pgWs7WAOaJlfA1JRFN6 /es/languages/applescript.min.js
sha384-3h4scyu8mjXlXZ1VtZqpc18Yfo2VK4icIjPT5VKcc6Z+B1QRvICaXfX4s9ICSpJs /es/languages/arcade.js
sha384-1xPG1kTsRkF3U6CRoWoLs4n4UOvAFC6opoueI1vWZ0GYANwTuOv0+um9ecHAG3qU /es/languages/arcade.min.js
sha384-1m2pjrOy/Dm/JliuRfZbGdoiP/XAXgZhTh6pRW+O+O0cJm+fBx2+PGqYQUBRJUGx /es/languages/arduino.js
sha384-w9lwucuIeYK8hzWrD0H1CZhvQLisv4eUmJc7pL7oMNkpBVpuuSDAwWJB7XW0ys+z /es/languages/arduino.min.js
sha384-dohVMr/cXV2TKOwUPL+/lGwXDj5WkStqxm1qggnRlXzCoAz5+E7Z29ExXaJt3Gp4 /es/languages/armasm.js
sha384-ocl2WOnIR3HmBi1ZJ2kfjMxq7rNhHxV1t4GFEmTWFMZLbHHXgdhR6HfCneMNCkt2 /es/languages/armasm.min.js
sha384-RUztCzxEjYyjTIr1xn04xHYn094/NrWUPagE08eCjI3ha3VPLU6wkyQpsmXEWyqX /es/languages/asciidoc.js
sha384-9XM0lQmJ4T2vT5fovmdMwihL4baMIxlzYPj6DUTgkGIQPNur/r5HVQmz7NufVMHp /es/languages/asciidoc.min.js
sha384-bMNdVcl2meGUeGsbj6h4ZgGjt+N3cYDKkgTHmNSP7RxFjjAaI0KxChobcQ77D8zq /es/languages/aspectj.js
sha384-IWEQTAPRfs5zaxhI0JynkX/u/7+0r4uZMoUGsHB0vr4RpLjuwW6qoHGq+28ZNvFS /es/languages/aspectj.min.js
sha384-Id7yS6/Kh/2HH+T1OxXrPHBvbvfMp7Zx2ez4w3zG0MGuQ8X6kWvti/7DEIUpb5dM /es/languages/autohotkey.js
sha384-cxMaZnqujtZiWcazO+5AIOVVliqyyDYRy8Nl21t796HxgHx0zOQTID4v/by9acdM /es/languages/autohotkey.min.js
sha384-Tfc3xd33+IkLS6W3EQ2JCNhgPgZDS5qLl0RTsNr5CCo+F6tRxqTAKwdGB03lJ/QN /es/languages/autoit.js
sha384-HzACo/Fj/mHoBXnxhCwNJnNAXGEE+r+6eUg302MQcRdDw9mVJJJkuFyWnl7Pl45V /es/languages/autoit.min.js
sha384-FDeS4TCFCbZW3sM3PA5wOmDbsMf3CqCMa3Nb0Wd4LoxZi+F7LUb+AZ1D0f5tKNqV /es/languages/avrasm.js
sha384-P5meR23mmKq5yLOMg2BF3KK/kka+PgB0U1YJ/Bn/IlzO6M1ZaRMeJ8Mq16Z8ujnB /es/languages/avrasm.min.js
sha384-wLveqgFaOQ9PzFODfn1IxFZ54KAIj/Opx/7gUnbN9eOC3XsWaK7zEAnX0c79VQEz /es/languages/awk.js
sha384-ZtSXWAZLFp62aXWYyynhvbh2tt+/B8/dq0PoZd8lNMmhGj9kec6R9TAkI2SPFKAx /es/languages/awk.min.js
sha384-W0sEGDKEyeWk1yu0MeKgxb6I6Re6+pg6UsZYSKD0HgFi9/w3fhzoqv1pUKV/GfC6 /es/languages/axapta.js
sha384-C8/VHu7qq1MrYknXkGgr1E97n47lEC3tu3i6pjOvWZgx1ObPWIYoO7MOvUdjvKQs /es/languages/axapta.min.js
sha384-no5/zgQGupzPFGWV8VpJzfQau5/GI2v5b7I45l6nKc8gMOxzBHfgyxNdjQEnmW94 /es/languages/bash.js
sha384-u2nRnIxVxHkjnpxFScw/XgxNVuLz4dXXiT56xbp+Kk2b8AuStNgggHNpM9HO569A /es/languages/bash.min.js
sha384-UfdqqRUCPXwjobj/0a5lNllzRSGiRXer66aOam42hP7YxJHL7In7sa0HPm2YakSq /es/languages/basic.js
sha384-4KNGcjC61b5C3mM3IHO/Cd+a9B/V8erOfOCkhFGQGV0SleDcX+Fx5u50WJNy+8rp /es/languages/basic.min.js
sha384-+xNRvcD3YWGqT7ciStBGqNYXdGXKFJscpMfmOILCxRouDmgEv++3H9G1vuHT4ca0 /es/languages/bnf.js
sha384-ZQaScy+UAXWHHTqErXqOiguzuFM9KQG3KY70cNyfXUtM+Q5OwX5SxrEiIrFuLYZj /es/languages/bnf.min.js
sha384-PJRu1G2QZe9Z5NlwP7YxB2gJaeXjnN9hNm7/kmZRKlu36zVEDeY2NtBSGNzWiuru /es/languages/brainfuck.js
sha384-tO9yiZ3yAyGDiWGcmVH+On5wYksAKu2TooLr2eQpIFEzasuxCj+JgEwfXc4Rd5Kp /es/languages/brainfuck.min.js
sha384-qimhSkVWof5rfaFajQk8KAtzSRYyIArcJCMKWdDcNq34F4uplk08wmEyUiYLmO+3 /es/languages/c.js
sha384-5fESKgrRcGs7I/89bn7NKFcHyvIVcmQIG4JfCEAV5Rg5VVtskrmGkHVOIsD1642v /es/languages/c.min.js
sha384-hZxqDLMmbstjs7YpIFFxz8ub5RuJLLostPP6tFqbd4TWx/BL3iZ3y6PI82giS2bA /es/languages/cal.js
sha384-yTXZ55gT3vqZ1Utfj/D4XrgwicCH36zfHc4Q0H+0WiaSOOTONI5unPZOsvecZvJr /es/languages/cal.min.js
sha384-VvEo91gqmgyRwrzbxd+MbuSPO7XEwNLGbAdKxcagDkRGL6/8jANk/3mjXwsw0FWq /es/languages/capnproto.js
sha384-7NkWzcNLtsHJrgq5lYW2LsWVHUMP3BojV5mpmKOxagVDi9V8Y+OLDT19sZOw18Q7 /es/languages/capnproto.min.js
sha384-nnLkKX8+s8kyExw6x1NpPg3YnfUqDJ2ynuFK7bmaJzy7Y55jC1/RJexXcR5WvjrM /es/languages/ceylon.js
sha384-xpVB1ReXdLGX2qOVBnOTgLfq0H+gUcc2dftOpl/RKha1RYSYyBYQ4FFKypy4oBul /es/languages/ceylon.min.js
sha384-w8WDm05SKpilNAN8PHsFgzYpqArWuOmTO13So+s7Mtv/y65iYeIvoywInp8JOAmg /es/languages/clean.js
sha384-1FpO7j5PcwDe8+fxjYb7fQJsfGEEDXe6Cf/BKX5BJB757pciR60wM+kwggBS/dNf /es/languages/clean.min.js
sha384-//XczpRyWOqgxCzM6D6yAyjGhLu6bUC6uIyzzcn2hIGnuw0SVTdPyamvB3ubDI5H /es/languages/clojure.js
sha384-XOcAzhOmbwUPkV0n4c9nRSfK4Ro1fPKehnn/hOns5fvsieIWnx/U5Czz5ljWIPgR /es/languages/clojure.min.js
sha384-pgAx9+rJEp9ZjtFWlHIozB8twvFAyCASWRuzWUVfRsce6DFGQZ7HrjjnDP3oro+S /es/languages/clojure-repl.js
sha384-pVO3LX5PlcHHL0Xu0V9NXHtYEkwIbhY9VijcsVsRIzBCzYvbh5ynInvcldfmQbxd /es/languages/clojure-repl.min.js
sha384-xjI3wQy5aAhTr4JT9bE8tZaEMtNn/fguEQOem1vG04xZl4Ov0uf8bsLkQUoUzvbo /es/languages/cmake.js
sha384-JdSkDmOvEqN4fTPk/zl0QpN/wGh53J+xJRFtOMdEwhZ2iDgT5a/Dl18e/TChDqjK /es/languages/cmake.min.js
sha384-Ez8wLO2kJgznnR3s6V0ubwubXwLpohcb+Ma2CTEiCIzWY3fu0kMVI6miw8z0FCKD /es/languages/coffeescript.js
sha384-fZRN66gLZ6Pkn58UJ93oApMbUf8HwDyjx+eP4bLuepyEegCg1DKbKOHHmHEZvHgA /es/languages/coffeescript.min.js
sha384-w4fUcE03Gul+Eu4EGzcmLjRgQSFbvZqIc1d42hzZLD1nkQ4OYj6pALKZ7+wGdrNg /es/languages/coq.js
sha384-YrTksxGF5q+81b7ZbllSjNa8awfGhHwVlnk7SALU+8jtDUAcEvNUDNA0KeUEyw5V /es/languages/coq.min.js
sha384-3P4f8c82IH2v/CRT+xNyjinbouH4c/bvODm7mF4SpYHBrcypPZBn/ParPlwbLYG1 /es/languages/cos.js
sha384-jnfP4KFzlfLE+tWZkDc/927s+A6R6Ks7wM4JLP1x72zUBQBoJSE48o7B/6r1kb/a /es/languages/cos.min.js
sha384-eM9Op3b0ilZ/iW7jeVAMo//MKcEXHCbg1Vf8SMrqds5LIOeF9+3qaX//TsnbItae /es/languages/cpp.js
sha384-+tDHTmLKfBxXgVksRhLEJM4z9PfcGQ2XsrZMDcdJ1SIlPZrtAR4+m4XUX+zJf5nf /es/languages/cpp.min.js
sha384-8+r5JgwvUtnLHBJw41lOfs/OcVWo2w65i5j4T8rtuTA/bfoVuLchqCxGZYm1nFr7 /es/languages/crmsh.js
sha384-BmL28rW1ggTPsUjUVLEN3lP+Vd5xub2ZRM83QPZHcU4m8bj3vzWMP30wvDWIygNL /es/languages/crmsh.min.js
sha384-GTGEE0btsNVN1NYLebyZ5dk9XT1/k1nuR8kgx3snXv856V13PEl2bxtW8f1JH/V7 /es/languages/crystal.js
sha384-3zYTwQ2p/pFsHA7rfbbRX6gggHPVk25IfZxXHsh/zCNy1YfDOlxBKV6YdkTy01Dy /es/languages/crystal.min.js
sha384-lU6If27eTyL2Yr+WS3ErF0/raeRKUheLuCM44IUaUshDCTvTeQijobPXY4wgkDGb /es/languages/csharp.js
sha384-k4z6XdU7qI35NxUF8SGumv5kMerrVg/xoat0iSaWnu/dHKoNZKdxZN3gI2WYgMfe /es/languages/csharp.min.js
sha384-LWvgGWXmngclEsDJmF3D0r9QTsAdIW8nZici4I4If/aw3c6268PitG+PNPAlVMFV /es/languages/csp.js
sha384-h92gzO66Hk1++bH+fXGmNay2Yf9WBsijCsnwubxdSE86gVf6UMC3drQ5VDjZaoaK /es/languages/csp.min.js
sha384-+9dzNYaLHp3OPspFCOJGrEwfiOV3yqeD/atUDYVt9zKUJ8IW2QxffCT2LfmGfwfW /es/languages/css.js
sha384-G44u1/pUATC8754FIKYqkCxCl9AQYnspnFxzuR3RB1YVnTvqOEofqvZNQMUWcY/1 /es/languages/css.min.js
sha384-IOR4nP1rYTl9FaQsJSp0RHS9aCDM/yF725E0GtYllv1G2WxGs33FgoM93Nx4UQHg /es/languages/d.js
sha384-guDR3VGIj675PAxqDTJxjM2Ew82al5msD9ySiJOzXfehlaoslILRHNUyrlxBSnZG /es/languages/d.min.js
sha384-XkaMmz1IWxxotnRaq8UyqY+r1C0SADoKTIT/8J84nio8ZUVtw9Xw2K+fBBvQ8ziT /es/languages/dart.js
sha384-XQur/dNXGWVSWOuMk+TsDoXfe5GzflLu972jJ72dqRV4N65jOT+h4GGpXQDstHWC /es/languages/dart.min.js
sha384-TPorW0AnOqfb0nnUevNZk4u62DUzbO/H1YjBxhFMfPQT43E+U9n7V1bN4R51yEFY /es/languages/delphi.js
sha384-KYBMbsTREEtzM5JsppWsdLqQsXtLiJkMauViGW5/rWaPzT0wDsU7yRHcjbBKmOt2 /es/languages/delphi.min.js
sha384-aZt7VNjEA1w4GeGPtZOPocoO/e6oIc9bMjNlN0GMXRkOoTMaEM2FCq4B6GkHANht /es/languages/diff.js
sha384-3J9ZKxCAysZ+DowS+TRZQFLDNVJwRq0pxq9t/JYsuFRmBSwgvJrbRDH4Av46yJft /es/languages/diff.min.js
sha384-dpaRAfrfqLJ/m4qzPviZ8UztzpU27eoBSH+8GjP1nkccMBZ6x9EnnhvlNYMDz0P2 /es/languages/django.js
sha384-xmWxMjSXleWbbNXqoQdpI8OkRJnRmiuJ8cmkovyFJSZxyjycv8DA4XuENvWPDLDk /es/languages/django.min.js
sha384-rHmCT89JQB59w2aVe2yFkMa+yxnKwA74LUHu4FN92sbr1FFDmYqv2QAwNBG47rkr /es/languages/dns.js
sha384-FyQp/yTYnwHmm+XuKfVGWxFNrGlgQ5KeEfWuNLOkj9AVIMPZPAx74qjwIQz2V5rt /es/languages/dns.min.js
sha384-conFwWhJOaJ8yLyZJUeX6IlE3YGpdoxazLFVV//QB5E+ASXMVyFnQE1V6SfgMzEq /es/languages/dockerfile.js
sha384-Dw9HMdbM7eZULiZ40cTZJ0NU88GUU5VQ22H+PVZ0IzxPGdnPPqKdsg4Uk3D2wbCB /es/languages/dockerfile.min.js
sha384-1dHcgFT3P6FR6RwYL7xelvz26V67kHZ4m22+9QEniiB6C/o6tAoevA9COGGaMnda /es/languages/dos.js
sha384-XSgDKTvGzlj/ixthrrw81gg7YE2TFYtyoqKTrwrQ6Eb/VIsbP8Krclh2BmNnJLfr /es/languages/dos.min.js
sha384-w4lnMQnIh5ktbhzM9nq2yQ0QvbJhXmIenyG9yP9/eSYMbA28Kjy87TK+jd+qZuXr /es/languages/dsconfig.js
sha384-Uc+MP2N7ERUL43nljo8V2Xtaw9xDiyS9EgFBrshaBEObPiqyY+IwIxzkkHsLM11a /es/languages/dsconfig.min.js
sha384-XEdbxzupbV1CGLS5hf5/JBCtBJ2gfI2YtYKDedi7hTb1BjK/5/7rE44dKs8i/Np9 /es/languages/dts.js
sha384-iGPPZyVXd7W9GgH6h9I03x+/J/rrwdsAHvSvvOkKy4RmMs146fYhCx9ywrKTBML+ /es/languages/dts.min.js
sha384-78pjAzkv5820KAdnWu9XFmkf7iBGzLpkNx/bpU5dB7GaDrvRMjstFafbatIHjzFj /es/languages/dust.js
sha384-7baMO61bDaNT3XOhMkeU9+88gLiAq763qqHp0olY9dgfDtIxisZo3LnoRrtLRsWT /es/languages/dust.min.js
sha384-YkFj9Vm7aCIYmCJRakwEpmhEt885TLf+nZCDO0dotFiTR/5XUa0Y4mz/ZkecNM9a /es/languages/ebnf.js
sha384-i+XFBJyknuWMcn7936ZYNZ+DDJ2mtBBAShqFgZkAVYz5ngJxMqatdUiD0gPr+4BP /es/languages/ebnf.min.js
sha384-b/W5D/qsSQTbKCPuKtIscnizUYtiwuhSIhPbqytRsu3sZ8hc5+KPeWbItFn9Ewi+ /es/languages/elixir.js
sha384-Uo+KLVze+/A9YCgMDJ4XtbATny5jt/ABQshKJVHut77YNUuS/zPVYAGKeLFdthjP /es/languages/elixir.min.js
sha384-HQ6WvsAcOVc5pibpLf67mgvlqP1UEcikksvBdv5kW9w7mP8WUYBQUl7WzKLCBNaw /es/languages/elm.js
sha384-hv9MdXKrd5QCWe9kkMdVR8bUwU6grFv3l9i5Qce+rRe56KnkTSGFKBeojjnMaOAG /es/languages/elm.min.js
sha384-+kBglEeWcSwdonwqsD/TpIjXwMn5FIeukuWgfAWv+QC1NDsZGeANBqgeir0aOENB /es/languages/erb.js
sha384-jKl1B10yci152fjeFFWOiIxAnyuqjj+GrabCB2d6Dq0jHIYs0kqsCF31KINUC9i5 /es/languages/erb.min.js
sha384-UprkYUPD60WsGmpla3Npim2r1f0q1YUvuLyap7OkTyFvaGCxjt/RcBDXFxGuFiWA /es/languages/erlang.js
sha384-ffehxKri+ZLFXqRZcDwKHI+OuCy3tWvJDjxmY7kTJNnF4EB//blTQga2gnOuh3UT /es/languages/erlang.min.js
sha384-Ks9Ra36zIOLujhBc23kvCUZzy1IBMsAK/rBZ2FrVZEQf6uzoxzu5+bWOhHOAEs8C /es/languages/erlang-repl.js
sha384-9HUVbJs7C6TSuS3lg1gYv9UONx7830CSmNM1C4hxswmRia0EkeP3w0VbufumsO+B /es/languages/erlang-repl.min.js
sha384-GCE+Z+7whzNpvoxg0r99r82QVQGunpGBsYcvS4vpuqEdyPseQRPnh6LTf1WSl3Gb /es/languages/excel.js
sha384-uMMeKU6xBWD301AZB6o7aBYDWroG+K8OUYDrA+cXf286lrDcciS6wBjzUnXFp1fS /es/languages/excel.min.js
sha384-MJLiKiMBGMHXEnN6ptYRW4oAm9IsmwQOe+pFy3WgJq5gu0uDct1QiqbZklctJgWS /es/languages/fix.js
sha384-g3uTC9nkSFSU4IGAEpkWH6/I3NJWikEbfrIpsA+SsOs9Qz3tYB7AoP97wTnd4ZDW /es/languages/fix.min.js
sha384-uN4Rg0qHAhhl51ZQUS4Y/Kp2bzLi4hxjLOoGyCjeOZyagpZrTDra4ElPNY0xUmMj /es/languages/flix.js
sha384-RrbUBKaPfrWutGiVHiEJ7EdCUNW1Ca39lVIM4XBA0ymj/DJ6CYlLcWqPiRZZio2h /es/languages/flix.min.js
sha384-Dsw3BhI8P3JC071/PDYuyCNVeWHyZY9q9mk7H5gUOHXaVxgKO4lhtfSuGOiLOxkR /es/languages/fortran.js
sha384-UkVn2T3qZDBm9TduPrIPBBINJQyzIVSJMGa2DRpLOc2zp6QZ3wA/6zUw190hnX9T /es/languages/fortran.min.js
sha384-8+x0somIBps1EZqOmOJH10vpzi+ZV5gWglDGtuGId33xzYozc4vqreUgzN/9Oka5 /es/languages/fsharp.js
sha384-RwyqgvGLMtAbdxb0JwJS46Hs0xvJDJ7ZhVOBb7vaGQOxJVPL9LYfT7FE10EgHGus /es/languages/fsharp.min.js
sha384-tsr/6YOOkdsc7sSiYgRhrONBY2QnZHOxdqFDxIHFrUvXHfylOlupl9DN1aqJl/vh /es/languages/gams.js
sha384-ZT7iPLg9oVzTWin6/2HhkKNkDNBHkwlxAAolZ4tmwLUD8Jb4EFhkyZIEnMlmLRDg /es/languages/gams.min.js
sha384-L2nStixf38QOk6U0Ofx4cjdXxY67+o15R1KPqV+ARscsKxYZr02gxjDMB7LVUcoW /es/languages/gauss.js
sha384-7iky+DInqFxLz45s1mmnohe42XafDC49jX2Uy8qT39RfxdNcrgtHiYe9MirdFQp3 /es/languages/gauss.min.js
sha384-IAd6YaXOquTp6vecI8sHpmYrZCl/nHZUYP4xhtogQCUpQeXUmKvNnKcppAga4sqI /es/languages/gcode.js
sha384-/crZfccBqGs5cAN8T8hFz+SYpNxD4xepOX5baIlqV0G8zaV9SuzTonApAXeiKy0/ /es/languages/gcode.min.js
sha384-PE5iEIjUC8S9zncEpxJjdWkY3650i/tGfXyZjqnns9WqOWzIvtxFfRmqECrJzYw8 /es/languages/gherkin.js
sha384-aOobeFVQ4Sux+80R0437f74uXacvfbsC01OHMCwOf9YqqqXdCv3xmdltL65Ztp6M /es/languages/gherkin.min.js
sha384-DRSBRBp34y5KixP9TEkoyASphUQfWiiqtSP20YveUIY05CjkoCG1wOe8A1cfMIZT /es/languages/glsl.js
sha384-0QXwzWESgZAQ8866ALLJEDZ9l14HhJTWnECXBxJ4DHDmywVn0LRIMqz7JJQWG/AM /es/languages/glsl.min.js
sha384-AqyXxpBKrtfMu0Xf0jHv1x6GDrEEdWDYYlPVAJYwgv42lw4DUFCgJ1FhFOGKGvGy /es/languages/gml.js
sha384-nz75K0pqPS/d8Q+HyuddJgG2PLLdsjnE1IX03ZlQP1cWx3uGZhHbgBF84cUfcx63 /es/languages/gml.min.js
sha384-U0cmcZmeG0JVcA3HKR6r7Sio0x8FtcXR7eviBCcgniMwCc+DMiV6IQPm7bFn6BPh /es/languages/go.js
sha384-5Mzx2XTmXU2XQ0AiQg/4HA9SbBDrPySZgpsOfSfflGdzC4bIpCjWSxIP62fOIFkO /es/languages/go.min.js
sha384-4tmLhe9PsyUKjoCy45+gSe+gND1BZWiD6jLvC5SmAK+GIPwUDx7IpzM/EhnkJI8E /es/languages/golo.js
sha384-1KwZxIzx78RvcZJCildSMOxijbCro6qqaPG/Wj3ustmMwgTYIiid0nenpiXqDxRh /es/languages/golo.min.js
sha384-nIINmygUn21VB68UmIWrtPRAzGMEgd5hOcfEEeODoG90i4mJzcACbwcWJ3wCiSY2 /es/languages/gradle.js
sha384-D7ZaQQrEs5BBMC2isU1LZZm9ITf31H0h+uPhGRdse290qklkkvt+WXc4sFh3m70B /es/languages/gradle.min.js
sha384-DmHzDhCE3ltNMxzvFexjBF+ku3PcoAMY7WFOXlWIZnV+vEiPek+T5FVXBuYELEAg /es/languages/graphql.js
sha384-y8dkUR8w2ApytztUtIg+27fJuiL5sH1oq9uRGEcpH9zcoYPU/ZFvroB77kwDE7d6 /es/languages/graphql.min.js
sha384-l7ODHGNr6otfXAqeIXIzYChf8hx0LwtxPazpWr8wzIkrXjSt8xnlIfxSxIgHCjT8 /es/languages/groovy.js
sha384-187D/dkURBQpq1omwwr2ia/1/OxrA8ok8uGZN7I5rHV0muATfS04w+mWc/kYTC2w /es/languages/groovy.min.js
sha384-st2jI7gdDuiIl9dQHBL5JYS7h83PuoMOFcWCAVm5JaRNJyZkoJsVePJK4PJ2SjrD /es/languages/haml.js
sha384-W3AwKM0k7ufGY1rXi/NC/GCvcGw1BQfGeDa7qXqV2IQbLoxnIQ0JojJtNxl33ybu /es/languages/haml.min.js
sha384-WA27nyafv5r7bEE03RHQyJ+V3X67dECBLXOy5MTKxPuBnbWuUBYq+Ke7qv5DkFPr /es/languages/handlebars.js
sha384-npJ2g3mwXMh9Py849SyQMEdEzTUaC5NsUuYXZ+C936ZpalKkv8V/XgbWzs1n2tzJ /es/languages/handlebars.min.js
sha384-rWXTmRoOwQzy75lNF9TBi7PvwMhKAK+S3iBleDD49aUH8XYtmK8aPZocifjskYny /es/languages/haskell.js
sha384-TaV6bIvb0PAdNigZuAtH9T5CUU7BCEJxOLBsVOeAb4O412gzt++0xyZtxbpQTwcw /es/languages/haskell.min.js
sha384-1laLqogt90eugqjenbcUN65mqzXlVo2k/tjQIpYxqiKDtYUigI/Gq175I9FvY1F4 /es/languages/haxe.js
sha384-qU/sDJJjc964Hn5j3KPIyChNS1VdgAQEeHQ/cjvYF0C255fgR//m65y9kdU6WF+B /es/languages/haxe.min.js
sha384-LscJ6z8/C3DUmNPrVYvELuEfKfggJ5IGpQOSNxgy1FtBlvda0tW9IAgRBnu41XtY /es/languages/hsp.js
sha384-X9TbaX7oKsQOaCnW/+tchCet/NzBVPbeOBFooLVGsQiqit+b66sUhevct82JLuSG /es/languages/hsp.min.js
sha384-yIW2CKaxiozMCGVe7a2RX90kdUjP0h2gALuNlfKbojKpQn1OmMQLGO7BOqhncFO6 /es/languages/http.js
sha384-j+2AgmE+4SlZjmviwPvbGypcb9BQNHQj043l9Bb3F2fnlusdNxxA5/INHsOrSE6g /es/languages/http.min.js
sha384-vuCI/QKdBMuxq2oa7f1u6jeh+4CrtgvYHBXymrDmigh8tDDiDYCfdDNAagEQ5Ax6 /es/languages/hy.js
sha384-G7+SF7XAzzELSBGtt76Sc3Gi1ix+fhD9ChAAvcduFY56+C4XGNqnQsIo4PbAjXru /es/languages/hy.min.js
sha384-a6TvdZS8mzqu0FTdhKbfeJJ30Z4psDxwYiDBtjHSv2ghTfj9cPwMvJFn+Fk1YYp0 /es/languages/inform7.js
sha384-8b3VeDRERFE/95Quioly3m60mn7GJEY8SOw8a1HMaheNvs9NJgS073+c9OaVurML /es/languages/inform7.min.js
sha384-0StLGSBIhoerTxrjwG/Lx1LYO/qmSp2TzCqgzCmnBDrVmkkeaFW9vHKuLHK5Ue7H /es/languages/ini.js
sha384-Qk1583V3PAnmXJ00e8ufkLJOuIZIrqrg5sTGoghEOwikzdWrdpiJv8lQqrURXjBG /es/languages/ini.min.js
sha384-HPVcHXUZLECWTio/Fewt/+4VbLb3FVR95iSfZ/ROxQ5mG7Mye5Hkr+bQcYpnCjrx /es/languages/irpf90.js
sha384-Rj8CWMElH9F9ZzXkxTheLhrrvROGyKtHkmZqCRmANiYQHjMWDbKQ04PY6OQVOxC2 /es/languages/irpf90.min.js
sha384-LZ0INpatDFO/yPz7qtJsJDJbADppfFUMMiyeVPn5caHg/tI2vJnrgOwt6HmznWzL /es/languages/isbl.js
sha384-vIQM0UP3EhcE9MH94YYzQE6uBcXu6nq/fKKiBnmUccpbatE/jkZ23Tc0K3NdiiAd /es/languages/isbl.min.js
sha384-ZCfS+s/zxY7O2bm2KoVJo1wUrLEpJDHZAi/LJAdJF5XjnfSWICkg6wHd2SEJGpyR /es/languages/java.js
sha384-716dHwZ7zbZcnEwTe7DLxlm5tH3Iyl8vSW5a2kYPgusEdp5k3A3jeZt0Ck+CjYE0 /es/languages/java.min.js
sha384-oQpcUGMBf+VDTHOLQ1uhPp1FgNBo0OZc9gbXGuVFwAogHlkh/Iw6cvKKgcgCQkmV /es/languages/javascript.js
sha384-3T8DJ91yCa1//uY9h8Bo4QLrgEtbz4ILN2x9kSM4QWX9/1kKu6UXC3RAbVQV85UQ /es/languages/javascript.min.js
sha384-Kx9GzIdT/krOCOIrNSOY4KCJBL593DZ50P+RAcbPkGLC6Bu0taQLuYTH7t8hLbn+ /es/languages/jboss-cli.js
sha384-m8qbMX40lQiAOaUTw6V/2bp3qnXMR2vYuOBjTm1DHcBQY5GRKisUsdnuFUR7k6dc /es/languages/jboss-cli.min.js
sha384-R87hRh4kF8+iz2sB6FvLrfR0XZBohjFXeJKIXld1Eji2UVi+M2+OIgJKma/9Ko6u /es/languages/json.js
sha384-QFDPNpqtrgZCuAr70TZJFM4VCY+xNnyGKwJw2EsWIBJOVcWAns9PHcLzecyDVv+x /es/languages/json.min.js
sha384-SyVlwNPJVCq/R0GgkfRdm3YeMOmpuNfm0T2iZJXhNkXHx0lVkKV2qoeDvaLZZjQT /es/languages/julia.js
sha384-yvS3Aa9ptrkJ7+gPUPKPo7dC6E0MIoYJQamXheN75r4LEbhr2NCHR3QuBfwus+L0 /es/languages/julia.min.js
sha384-U+LMMniAaBNBP/0WPs6uoS9LqifOFy1QD4Nbt1Hofg6QnHTMAbR5fnbXEj4GqUms /es/languages/julia-repl.js
sha384-19YzhrU5CG+oV8eJJVC64Hir10cuNEEEvqj/Stura7ZqUqbgDsldpSfoWunbHnVO /es/languages/julia-repl.min.js
sha384-prYrGnTm7oh5PuKMqLmR+7SrkN+R30qFCvQLyWRjl7bJqveap4Mb3RDeCe84KHSC /es/languages/kotlin.js
sha384-yM8aIXHTDq77S7Ar7r/O8Ix2yH07cPC1z48Qh6HBzcG2+plE8YMQn9goknWvawaH /es/languages/kotlin.min.js
sha384-fbxNoZZaPZs3SreGCgzOe8ePQoyxS1dLDpzyq3N5Jyaa0yMu0a2mWHm/EmbtYqi4 /es/languages/lasso.js
sha384-fvrCkD6kUH0Y5flDk5fM43/nR1vGkZ0HzIoNLyT3GQsRZFOy7ZWY7lwwjDS0TV26 /es/languages/lasso.min.js
sha384-qCr0xtEbtIU6E2yKVSftA08wjUBxW2zlHyVvXoQoontrc8eZtFMO/vSWWFtJhpzK /es/languages/latex.js
sha384-Nj46FmS2fPnUrHYhGDrVZjS+SqPPtCVnjiMaPB2WFMnYAWV8lW1T2EQ2gxq8gDs1 /es/languages/latex.min.js
sha384-USLjjdQta7euVyzK9OK2G3xKBEFBllRrJRlNhIx2bjVwUSx7jA6Ik72tMUbHadaH /es/languages/ldif.js
sha384-IaB8ennafOrABIWzfmPoJLQ1Bnd6HSlO8sXjy4lXzu4rPXbGN9ul2o+a1omLUqKO /es/languages/ldif.min.js
sha384-EOlA7X6MKRjIpGYx09ecO5jAMoNmfB791nh3OQg5iEZgNkJ7i4fDZUY1JHyyt38Z /es/languages/leaf.js
sha384-zSVAwJVXLuIs4ZlAQiIC+Cd4cEMWt6w02VRWDhzVKPHMdmzSHKb6xTRgQcuvXc6X /es/languages/leaf.min.js
sha384-BlC0mPZAwpk9QZaocgajHbYH8yYkOUZrDKpeaPBeCGrQfQ6eTZXyInS411HpGnO0 /es/languages/less.js
sha384-vmEaRQjBedDEQk0PmHXdDeC6kbUsP+AhxtGkI2jQ/VdSRZ9ndnDOBUuxp+cZ7pQV /es/languages/less.min.js
sha384-0k/dp5w4RHaCfU9U1Uzs012uUnGBbTbDHnrce08L+mg/OoQO0xRVv3BDZ4YvhzHJ /es/languages/lisp.js
sha384-S8uA27fsmX6Fx6EhK08sAFvjkDhHZSMfrFbNgqDDsstYlqK8NVqKK5TqYldxjDny /es/languages/lisp.min.js
sha384-w/cP6O7RsSPYgWefN2NrUI2vOH0Pnz4+JRnQdMqAqp316ia+9rzFpvsiqdlbjdVD /es/languages/livecodeserver.js
sha384-qu4HCOB3caPjcm12Aq1NoxdXwJNKp/Doig00wy36re3AWCpnG1jl5H//vkqBJk2B /es/languages/livecodeserver.min.js
sha384-//z1izN3Vn0SZqVj2kdPDH2+dtKjsNRkw6iI5eJ+oUO8sH1rowQShBQNAe1AmIAI /es/languages/livescript.js
sha384-P0AzyZPotdKwowy/WdkLtC0E5/ezqsShC3N2Q9q4f6TIB6Hn3vxIOCyXAKKyJyLK /es/languages/livescript.min.js
sha384-EUUI6yUHTy79oan3YmaO4JgOeT6F/YxDDPJJSYcyhtSak1GZiPjsR4F/YC5getIP /es/languages/llvm.js
sha384-gpuZNMDkXtGAGcK7Oc23L5g4M6m2q3cDgegXlnpTN7euamQLboWR9c5lPIrINbrq /es/languages/llvm.min.js
sha384-AILzYjgAE0qtfeGrNX4h0XwdrFZ3nSEcVpPt2iGhG7WBI48xbRnfQhkuMNZzYbOv /es/languages/lsl.js
sha384-UwHRKsg/zBFMdfj6qNa9tgf5dOEgf/9aylmmn10aghnjWhWKzxSqRLvPC4tL9jAT /es/languages/lsl.min.js
sha384-OJrXTVCemUa68axskWVxZHXuDfyPWWFK0X2cJsPaMBt6jGY1HxnBZJVPEHTqou4H /es/languages/lua.js
sha384-cjlU8DPjZ3XY/dLzIx+CaoB2jzKXf43s2KSU2rZQGmxwR1d7k9p+SFt1IsNkFnnV /es/languages/lua.min.js
sha384-npg+R4K6p4Q5dEzYDKy3gZ+l4mGV8hDMErOZdSSvqLxED30Fhmgb54WD4wkeY5yh /es/languages/makefile.js
sha384-Ev1PV0+HiSwEbi0IfJYmpMoxv3E0sWhAALg1frIiitM9zh2BVDe871H9Z/RGXqFM /es/languages/makefile.min.js
sha384-JkFMmKMbHcXjdfHauRnREGG6i73GyMisdqNivBm4Z9m2Oc82YIu5jQtIjLa508e8 /es/languages/markdown.js
sha384-65/lNNIY+ayhHTtFznjnyZ5w2NDdZIpSEcqjss/p4ex5JamZ46FdYwDDf6IBLCmE /es/languages/markdown.min.js
sha384-+k7Urw58KtesMajdnc5oFyHaVKkQpQuV2GseJ4oGD2nqLfkMa/SAceTDT1xxxpvL /es/languages/mathematica.js
sha384-NrQRnRP+ItgqOqxjc5lml+y913cYS5pG+mKH2P0J342fuZ21EcABNNJf547J+Cg1 /es/languages/mathematica.min.js
sha384-1M7gIlbX6010yChY3W8wnK+CzvS7Pb6uUdjB786/b3Y+KOo9Amkkp/n0LEHPtxfT /es/languages/matlab.js
sha384-wNVkpFZOn59qnU1wCeePf9PzQGAA1JeukodpxeRj5ldSPyLeLU9FPMTKFPcFFym4 /es/languages/matlab.min.js
sha384-YUvqD+p8mokJ7RJUQaGofAFyry3tcOpWyRJSUrsMNFDZQqdEU/tgsCufdl1GyFDe /es/languages/maxima.js
sha384-sLZolVMgRf+rWbsksKXmI8jcmLv0pLNgsJHWxfegD8vxxrtepmgxMZbHg424Xo4X /es/languages/maxima.min.js
sha384-V9eZnDeyLEChcdw2S4ZrzbR+FjJOlFK9w12EWfFWggNGNQWiKFueS1W5ypgy5Us9 /es/languages/mel.js
sha384-ktem8uFt3dpR7cjM5U2q/rcOhjrg9RTRnjRvEN2Y3X7F7CO2/NI/9fVNhUWqL51+ /es/languages/mel.min.js
sha384-5gNZ/scyDXbPmd6ljT5YJ/sE0hJGWlNvjaX45BFScl5O66qJbvZAiPY2vTV+oyCM /es/languages/mercury.js
sha384-L2bh78DiQIIiXEEyhBU58rhag6CSLrSk9ezrNrTIdXlvzltGVPqN0SBPA8QVIchm /es/languages/mercury.min.js
sha384-gMzC3hkIwZjphoJMIhit7PxZx61AJssvwGFe8vEDLw11n5Oglv7+CsYT0Bk4U3NR /es/languages/mipsasm.js
sha384-hzQcVB7Qny4YQPUuHQtZ+wHvcpHWNYimKmHONk5g77Fca1MwwkNixH7uPmFnZSIB /es/languages/mipsasm.min.js
sha384-oSKeRRG0BQvx83WAgb33VVlWtBXVi0MdQKnH/+eBW2VsYW0U6UBu0ZoML6fusjnk /es/languages/mizar.js
sha384-GMt6gh6Sam0b4tO3URi3Aayc/mBtERcOztn8gNiizdKCmWKcZdBKDa7VOKczR2T0 /es/languages/mizar.min.js
sha384-wHi9YWuBlMmp6CUES5L68AMz2JrahIIwLinNiyoYpYMTbvOs6HMSxcSmkJk6jslj /es/languages/mojolicious.js
sha384-riyMTSZ7x2BtezT0NuSUcLWe5L4yYLXp2QWu4NWdo2kZOs8Y3iBTcd3M469B49Qh /es/languages/mojolicious.min.js
sha384-GBJkYpI5k+1/BqIWl3RZ5zRfSREWoA70iBFNV9l/NtBrxCRpBilE0moJV+svIxBW /es/languages/monkey.js
sha384-yNAYZJH77tgJ42VoPB0eZbDd3SscEe/QadDk4z0uLMvq2Y6DV3G++6XuP0I6bh9S /es/languages/monkey.min.js
sha384-umtgRCU7BzOl1wxPBLLGfFfrj9eXy6O4oS3dTWkGSxFLoPF0ZlDgt9uC2dkS/zK5 /es/languages/moonscript.js
sha384-hXY2GFW+d9dPGMJ2+S4MTTRFSRE2QstWS1MWWDmjNDc0FOrcC5fNbr90mh8awwa6 /es/languages/moonscript.min.js
sha384-Eh0spGrv8SvNhR/DqNzNA5xJA0uEy/Uhpm0pXMh1yTMO/C4u/HFb1DfQS7PXh5+3 /es/languages/n1ql.js
sha384-qVnCpZxP0vME5fvdyeGfdfPu5NbR3mSIi7UOWNGtON/opL6OfVYYSdK1u2S6rB8P /es/languages/n1ql.min.js
sha384-Tzi0ytT1W0ybCHvIzLzUAekWIQUH3FC/yFHiST49HgYHtbrUfF+MDqc7lAoSjz1L /es/languages/nestedtext.js
sha384-VY5MjPeVE5QrSnKCEuQlNC60blB6fFELWFgSoeS99bkXap5NQw/D2PsurTmoZwOd /es/languages/nestedtext.min.js
sha384-WuxmFqZ8YXr3xyK1Salq5t1q46F/VyzVGx48M2ZJPhbodzQ9L8kfnP/0seU78NsC /es/languages/nginx.js
sha384-XOua+gbAwDawIeMkI2pkXOZH9Xxl9/XLoGuPJD5Bs3WS4bMn207o20s+aQtpsqqE /es/languages/nginx.min.js
sha384-fkS99JygS0zfe5l48XPUG1OXCXb/4w97xFWgSvca6oV4UQIAL0RZXAkNRHso87F+ /es/languages/nim.js
sha384-2xkAL6oEvjXQcavQSUBRR0o9ZmaPa7KtykaGxiSp7Sy42KpSCsG4cvIpIXk3pJHJ /es/languages/nim.min.js
sha384-ZpNIK1eXiTo3E5ufcyPhbl0oHJb/RqGc7j2wo6XeDeASQ3o6pYPUSDdsvEqHk5WK /es/languages/nix.js
sha384-ZDRzqYOSMHFfcbhNtV0wdJX9mMQ4GIp+km7Re8Dife/IK+k9BqYeoYWAS26Ld6IQ /es/languages/nix.min.js
sha384-9gI0M165Jidq8eabQw0APiZDRGdil0G5h4l9HNX6N1Ar3yInmbVPF5su/LVRqowo /es/languages/node-repl.js
sha384-WUouP1qTnssBcr1Txhf991xYhbMsu29J0AeMm8Hk0Vx/jBKehRztpDrtLV5chOyH /es/languages/node-repl.min.js
sha384-PgHmsLaBGppH7l2MsDNXB2/4m5Om+vI4uSbvZQeozlf5s/p7dSw4cma+taF/nU46 /es/languages/nsis.js
sha384-OKDl+BTp91z+xOwY30l5qXV194b5FceaKoTbqgMAEg9Sy0mhWN90wTJeuJ7IvFBp /es/languages/nsis.min.js
sha384-kkjOIVaO1cseOnYgbgXx/sE1dQ95zQmTko63ud4f1DKwHC8QwzKGH2B9N6w2o9/t /es/languages/objectivec.js
sha384-hFyITHOZcjIH+h44SyQkx6EmN8ato2cZ9DaY9N6C7jLs9dNEmZrLKcCKI50zvOSm /es/languages/objectivec.min.js
sha384-NkACruoAWMxpbP3nK3uWg9ZkJo9tV8u35exn0RHpUEoG2Xk4e9PgH6YoFXx6SGzM /es/languages/ocaml.js
sha384-crG9/szkNiDEO5kEofdhAugXK4heD4+rVymcFWdMD2zju1jMnmwj0+zRbryhqS7P /es/languages/ocaml.min.js
sha384-svW3oY81RIMvrP6me0NU9DdIFsDSOfd0GVgGSZ0GE/keBJb79hQQBRaLwT27LQ0J /es/languages/openscad.js
sha384-Nm4R0c/T+nnd8HUuYtwqTIhgYXMPvx2fzjIQB30fzczD4Uk86ljdbr7GfP7VgSOO /es/languages/openscad.min.js
sha384-0qePsh6WOfFExmAwUjW43Q/K1FiRfc0Sd7SkZd6tHMPYj0fyg0y4wUlkjx2fqr7k /es/languages/oxygene.js
sha384-y3Xa2254pmPcQWr6Wljhj2sgqJounTcPFIrMkKQrWlWp7zzJHO7EpPdU4YxOVll6 /es/languages/oxygene.min.js
sha384-q/J52OGh4UHE/U8R1OxK2eeCwgojYh7pllFkG8nHZlhjcu//R92hgbDVLLFvLY1H /es/languages/parser3.js
sha384-gXlSFjAQdfSvHNnvsQzb6gF3Tg2XcgxEoaoJNt3eQM/zYp7bbECyesWhIDCim3Ro /es/languages/parser3.min.js
sha384-TWMQ/3YwBMm0b/GhDxqPJHcRh82R+0fiBA85TmnqHGfhccDJMkueh/BQqfOxlgb/ /es/languages/perl.js
sha384-2CHs1KY5b+PYxl5fEs1H9IwHwE8eglxOgjVwD9dlhDWZfWKj/w3uHtDDH+1P8nYO /es/languages/perl.min.js
sha384-HML+wSl5DkVKzdahoj3yemQdErWxe/nxUs+A0fVbeSmFzp3b+/NtkdYL35Nk+8sY /es/languages/pf.js
sha384-WeABEWvvGME3DkpaC8x0980x/hDPRS0Wb2w4RkSlEpzNfBjK1pBJMsziSkv6+SzK /es/languages/pf.min.js
sha384-+ORCyjxlNVWQ/xCTFlWWc05iwK/J1wTnRy61YcExQSzVjhYQj3DPGtydA9IB3KQy /es/languages/pgsql.js
sha384-OH8IAQNE+/4/z6sdDTBio6BftdgqOsadsVYkSooVoUwdZYg95qly80XMjUigH5zk /es/languages/pgsql.min.js
sha384-BxojDi6ePBYN3unEc6aUEpUtUyx0Eq0i/UZPISuI2YQy6eAD5HzD0dtBC53uZ6R1 /es/languages/php.js
sha384-C28kAnknmyKQUME+2sa0tyQkFMN1S/aUSB1gQ+PKUf0+ewgxNh8CwAmd+C0UUaZ7 /es/languages/php.min.js
sha384-ZykOILPkcS6gVzXO56vF9mLmUbS20XhOnzaf7jxH0c13+7mKElNqtP6YCpBeND6v /es/languages/php-template.js
sha384-xcrjtkX6DU8emOzX0JkgBdUnCB6E5abIF80h1JDwLKDJPzmIJSWyaUELrPg28SsM /es/languages/php-template.min.js
sha384-dkR9Qv3ZGmcTGGFP26gmcHC9DBgRYE0XLGxF3mBXlBZaBrscW0vIiVN7oTyQmrbe /es/languages/plaintext.js
sha384-AkqanemYxn6S3BQnW2++1+xqywaq2bJfFlfiAkPNd7Yv5t9YsS8tFzVVopyOa747 /es/languages/plaintext.min.js
sha384-YF8jmdz/83z9JEBjTeQPHk+lxd3/pwJEiPDhL8bC5VxhSLxXHF5avycRh+y2IJoR /es/languages/pony.js
sha384-SWQ8CV4qeox34/B5OTaJlcCHnCUyGjTzwB7ZB0Z+VmjtG/1/0DK/XXk80l8/duqr /es/languages/pony.min.js
sha384-8aLAoO/YSb+fhxwbMNNuUeq3x/FJJvtLwRq9nYpoyy4j85IpQURkoRySqJHBFlwu /es/languages/powershell.js
sha384-LA+GHsDHAleSQDZ+9MEiJmTnrJ392Nkux5b1RIiGmzAkRo+4dwKsjS0z/pzCXIFr /es/languages/powershell.min.js
sha384-YhFioQq+M459546JNsJ++7mn7+6ZCmkRbUPGL/YBnhckluv7ioTXlsN1q8JV9Nag /es/languages/processing.js
sha384-8KwenjihmQdgtsLsST675iXMgQbH8MZ8M4yyaaUY34BK27FxLwO3t6Vf/XnrTuLN /es/languages/processing.min.js
sha384-NY2tVnb0euiAg2wgksOOzgJ4gf7Ao3Pl2u3GiwH4IC496VGGTrS8N6FsGL9SGQlM /es/languages/profile.js
sha384-pIXuqck4LfmmmkzhWmHbUMb1AHwLVFuI523GlUYcuNZ2X+wVdGyB7x9erE0Jeerv /es/languages/profile.min.js
sha384-iU/MrQcZQ0z0EJpYYXVkYZJR5CwypGp/ZT7CZLMNhBzhhMjV86r/fny2j+7dkl/G /es/languages/prolog.js
sha384-OzCC3CRxx3WbANdD1Kmda4no08jqEbOptoYfXKZGrDWQ+vwLK5eHbx0r3AeMQkVV /es/languages/prolog.min.js
sha384-Ws6d8NiJ3KKfuA1cOK/ZsY4zuyoJWeM2RQpQFv7yHr0J2/5NysTZV9nUqvFGNUqB /es/languages/properties.js
sha384-igG9bugojNt63V6EXJUpd8a75vlEeh+G9suHO7+lk9/sbdQ70Rn0KYsJNRcRxJDC /es/languages/properties.min.js
sha384-AL2CG8F0fSCEbgf1XBvyL2owO9Iell6Slk6Yque87338pLjMk9+22DGe2qDic9wj /es/languages/protobuf.js
sha384-S0SG5LPfLffrYWmoL26SJIw83wbqXQz7YzSwoYH1Fv9oYhT4tok23r4XwLjhwdot /es/languages/protobuf.min.js
sha384-+pgnmuQdg8xetP15uG8KjCZ2IbIxwSL2eMvwpiZ79piTT7w5o0hJudz4w2CtN3Uy /es/languages/puppet.js
sha384-/piqZSA5WyjAU/0Qsz4/SJqEJY1cc8p+caPB7I6Vt70YqB7wmFBtLZQIPKHsRTcO /es/languages/puppet.min.js
sha384-Yh6H7nT6lbF6Zjue41WP+MIFLsC56wXWd6ZLeTr1X0tgUDrtVoZBaufhOBYWvOIU /es/languages/purebasic.js
sha384-BWESDJxF3tlHE7zC3x1g+vFLrvXtfyWcmOnyJ5iuaVdJzysOe/C8Rl6fii9KXKYV /es/languages/purebasic.min.js
sha384-e+d8RFZbtc5Pmt3xfX9uuElm63v5qOj7T5hAkkFbnYc1wEk7wCLlzOsm66MCf5Uk /es/languages/python.js
sha384-CPHh+9FxkWF3OtMZdTj5oQN1Qti0p4/5XBABz/JdgssOKHmfAOFz6Gu4tsG6MQiH /es/languages/python.min.js
sha384-v4RmzEvafbIHjO9uW/veEOHMjdo+L/JmBjiMvDQnib/V8YygI1VCW3uvAKnt6TlJ /es/languages/python-repl.js
sha384-RE0OsKNDPyfYMmPfLkTxJXJj6I0NAc689xyUOr3+EcFqTQSaBqP2TF2UPSY0Qpj4 /es/languages/python-repl.min.js
sha384-KTy3F6FlTG35GOKwmIJQP3bX+hwywvCZwtFQfWwDzHGIAe3ZtCbmvzQiSzHymTSB /es/languages/q.js
sha384-ipG/kK3puSleu0UZUpK0ddy+AdAc838rDe1h1gTse2tLdUq3HSlK2V7PIzZmsqQW /es/languages/q.min.js
sha384-ikSUPxZJlcI+3uOKeWQpy+BcwQcYiQbx0+/N5fb1sd0UE7TBOC2tUb8N9C5FLPVd /es/languages/qml.js
sha384-kYUxqwRpm3RkSmBRGE5WNwR959eF2Gq1+oyEE5twaZ5PfZnLRwTUv/5DwIpEwItv /es/languages/qml.min.js
sha384-NtjW7lI7c0f4eQUAsyYEvuUEeZPVwqBsisjGqqXls0p3Acc2wX2ppJxBvFC7jet4 /es/languages/r.js
sha384-+SoUnimzbx9Uf3b+zsNgb58sz3J3nRHE7ftjhuB8hvcb3K8D8G0Fe3Bu4lBrA3Fj /es/languages/r.min.js
sha384-I6H9vsQbeIhUkjg0pXV///uZZ3EnmUDLYY9NtcnpjQYDkbFbYNwwLt67pOdWDn9I /es/languages/reasonml.js
sha384-brwFcYLUPw5tzs2aTT7QpBBDJr8KOM0d565m9nqmZVnkGOro7yZiY1LMVkdd+Cz1 /es/languages/reasonml.min.js
sha384-+m9n/kKanrxb+bcsIb1RGy95tZ+H79GVfj6sSFdNhA/gZoP5+GZY5tpmiXD7bEc4 /es/languages/rib.js
sha384-/iToU4LYE5K1+u9mjsp4ezWGcJP7+loe1xp2GQbhNepeNtU1a9y1bC3V/hjyP+IC /es/languages/rib.min.js
sha384-slX6z9Fnx93Prkd9/Frq+BK7D3PiFQDKa3+drr8tjls1l5RSPVM8T7KJAmeKpW22 /es/languages/roboconf.js
sha384-U819GJWTJDz2catnyvYguPUtCVda4qeZQ3y44GTRlIZxbThKuG+X90YNPN3VA79q /es/languages/roboconf.min.js
sha384-Zy3OEBsiwlfi2Gb6cD3KV5+/Cm+o4lKdpVNhx+Ex1EySbicIXK3hcgEoldbyQPOD /es/languages/routeros.js
sha384-zYk2sUFnVptEfXlETVtcHjgCsqnwsPAmiFluI+w6buaS3NABQ3A1+wrfLbpCL9wF /es/languages/routeros.min.js
sha384-6YkpGIxhmTvceWvNhd6BfnOl46kg/1G6RsQTIxKO1Ofypo7ihgT6B57Y/RMO0Q9f /es/languages/rsl.js
sha384-TTaTGhHJtVCiuponl3c3W41yk7oceUGvbthSIf0YIAy1Ua89nyjf3/xROIRa0ZeO /es/languages/rsl.min.js
sha384-Blm/RpTi15HpdenfAja/zwDDJ6bBmIoAsFgHcNpw4u+DqKFZEpB67DIa2A1NXtZf /es/languages/ruby.js
sha384-INdPgGNAH51T3uWXoaYVa0ag70hxlbvjTlhyLicF3SuuG0BVuycs/GrFGi7gt/8a /es/languages/ruby.min.js
sha384-6oqCVtUXR5bhx5bmDjrkk6roZzeEy/FUr7igNx4JcRA4qkPxER2VG78sUVMaGoGa /es/languages/ruleslanguage.js
sha384-2cU0Ky0iglRofoEhsHFwvU28aU2vNqby7FLYyABx2Et4CSHg1SMqT59fBIOdhD1S /es/languages/ruleslanguage.min.js
sha384-LZ+YGF0Xve9sHzC9G37Kc14gDC6lfDpny2hggVJVfHb+OsTEXgMGLmAWUJzi4YRC /es/languages/rust.js
sha384-/ktfWRgwL+kZAZeeXDl9mwkD/3atjwjkzLCCoSHtME7MzP87wMhUmNUZ83AoqYx2 /es/languages/rust.min.js
sha384-OLy63fVlveaBES4NRTcZfIj+Ck0KifxzXsq78wjMTc0zEkw5KjybTdIYl7B9gVg4 /es/languages/sas.js
sha384-VwBoSEidUQdvDdOHc+Rxj1gak8QvAOq3NMw8vBrGXqR8HVTshFM9DLOZ3a6Xki5h /es/languages/sas.min.js
sha384-bJFk2tsVn5F8YUHsonGtmJRBQDWY5KfeY1yoF/KyRKPSe27EXGZozBtwNVK2ZBE/ /es/languages/scala.js
sha384-Ww9na6pFMpl0LFINRDLDfPQf6xtS31SnqKhTud2UgsqMjEYdiU0UCoL6aNcUg0Gc /es/languages/scala.min.js
sha384-g9Nw+jcUP5Wmb9MLU76ymKmuokDMERuE6HWTbZAmO+SOuxV0yYYCzFX5GKLvMGDb /es/languages/scheme.js
sha384-sODLwd9vR209AK944nKCuMOHKvQLBtDc+6pxw2Ofgc0aPWbglehFBTIYya3hqakl /es/languages/scheme.min.js
sha384-T3tV1RFwkTJPhPf+3/QF4Xgc8lrEmXK3Y5EHt2ICC54ALrextQfG1rR5gq7/r5N9 /es/languages/scilab.js
sha384-V5+UMZKcI4a5pMwGEj/UhnlXoW42mVpPU/8bZCoGiCZS+96s88J1Vy7Ne6Dx97NE /es/languages/scilab.min.js
sha384-kcAcxYNWYVMr2nwgsILhFHZH+OkRjbK6xazY53sIcW9bta3onTtDBKug0swjsiMC /es/languages/scss.js
sha384-mlx1v98SAsuX2QnqD9+dsXOfilvT7QQjfyOD3HTrvA4TXsxV4yjW/+TqhUADUuOH /es/languages/scss.min.js
sha384-VYwyP5ddOUunx1AGpbtE38OKY2PbjW9kk6X6622tvqprRJk6W8/tgMvI7MqaOZZw /es/languages/shell.js
sha384-gRgqaIgHrAR1qlo866rkV1AUahqopv0dnpFFLAC4X3jYTDagRCTxQBTMFGV1+x1U /es/languages/shell.min.js
sha384-b+TcQzb5I++yEAOaxeivVDPha/acS/KR6FVBY7ysx5lbHU8sWXktCbLTkVpeLLZD /es/languages/smali.js
sha384-fPEykjaz8PnjzAA8A14l3GD0L1xBV5CIr6Zj2No+5QcdbKY9qkjAOzV3lTHzyUbL /es/languages/smali.min.js
sha384-pKzgQLfmC8/tcjH6cfKFH/Rkm1z5EyglnhZv+Pd2UCrFoQGCiUosM1DTlEIS4NbL /es/languages/smalltalk.js
sha384-uDAwcPj8hOxgjPWTW9AQoNeZKeCtfo/e9dBHntXD3L3KpY8wRg3BgmO57iLd4ZyT /es/languages/smalltalk.min.js
sha384-UX3q2PLjL3g0XZtWhAnZGs5FZWcKmfsTCSL3YpKaMku1BQiL+sqbA34ksr/23zbL /es/languages/sml.js
sha384-tYQNE11b91fF0Yxb1oIax5gYF2FHibKKcMNwvFoEmgewqH6YWeAiLHe7VJiAb+1q /es/languages/sml.min.js
sha384-SSvospF/l7n/2RUsyG+wvyD3ZuN+Twy05bsDqjA8vaPJYALZoh3yZX3iaw87Zd/h /es/languages/sqf.js
sha384-cTYdWLIEJyEvihgV6KuwAmH44Rvisp+2xGep35NuqtSukZC9AtiV/zjownVDD/a/ /es/languages/sqf.min.js
sha384-ZX3mm6sjLYWMBTMUJYzvQXYHNWVJkD+t1ppx4BysyVs6cVhvYFVuwMlVCeAwtwm9 /es/languages/sql.js
sha384-DloKeCkj/pTPHeqWu3keGoEPpZJGm44yQjSmSfpWasykAMUobM0hcYFFPsg1PE6K /es/languages/sql.min.js
sha384-qHeIRIgd1onURzbRMsmjReo+ilXsCu/idtDL30db8rQ0fwZz5ZKnZ94226CiHqjB /es/languages/stan.js
sha384-6HVe/DzulN4+JYhskNbIQjIrL54HujHvB8kkYeroH8llfD703+UkYbSRcoV/Q5Gl /es/languages/stan.min.js
sha384-10YVbbah5WsGByqSbqiIihp03R7dbfuvcOjHMbX6PuFx6CYkbdRNTPRQzuuQCTNP /es/languages/stata.js
sha384-S47i75uP8zrBJ1LgUUP540uu2oQOVjSqqxk9xUhiLSGTsaCOzNdzNTsTBM0KrtMv /es/languages/stata.min.js
sha384-S1qpABZqDdXjNUDisbfr0tinyHtVHIY+a0+F9v9A8JJd6ArW83vRfa7qz8NwKy17 /es/languages/step21.js
sha384-LK7AZp04o1ZCkbd4H0RCrNpbgt2Sqtoiyy5XFtrxi1Kl1qsemyf8dJWSP5Eg1vpp /es/languages/step21.min.js
sha384-cu+ZTiz+DPlJzlpOCZzjhzj/sZd/xzvbWzjBP9QALEVkQaYwToO+oepMjwImUu+G /es/languages/stylus.js
sha384-p/nPinGn5chrt4cTKyQIBe6+Mq3I9wWfqtxG1zDeE1MFRpZgR5u7OMwne2uePVM2 /es/languages/stylus.min.js
sha384-qB8xgCQLd6htw+fHtV5/TJw9e08/fE/m8Bvw01tbwAEJzoqEQWSBlWsX2HJDmVwn /es/languages/subunit.js
sha384-eljTpkJZPyqNx/Ey5VJAmULjWQXdW4nf3Hw796BQza7d2zZjPeGhkAVAweLgTwSG /es/languages/subunit.min.js
sha384-5tfPU5gS6J1PWMpLzFJzYt4O09dEwu0Ie9HPKYUQ43EE25lVsyugx9jVH0VViMor /es/languages/swift.js
sha384-a+qBlyBQMw6U8FrlXL2OodB7uR5S1diebujSfAM76WdYSl3ydnlxf8MCpGaiGgWG /es/languages/swift.min.js
sha384-iuz7q/jaIFVBg+asZ3PZ9f90jTiINnQ9of1yWxCMcr2z9MRG11slSMo88LEPAwjP /es/languages/taggerscript.js
sha384-YPX2/Q71sJT2APyCCwlDiAoevUjzpv9PQQtucLfqN65dDVt1qJOai1nW75AsJcld /es/languages/taggerscript.min.js
sha384-GJ3oGwwwxor71guHvgt1XYAqafOT5haO+45kpULUG4v6rEYAme/kadXtqkMBJy7t /es/languages/tap.js
sha384-ThEjd0cqeUoMo1okPph0zKqAkp2PTn7lSRyVO+Raqps94ti82wYRn2F96Fh/VJC2 /es/languages/tap.min.js
sha384-Tj3xAByOtomCg3XtxU2dYGw/3sBkYFbwbSgJ9PITLdWIhqAhUOKXEyuWKj82m37j /es/languages/tcl.js
sha384-3jRAK2MRHgHVTGdcVv6Ldg8/S3ObC0JTg6o0hPnMsE1gRHtUeHTC9UEyLajfWRh5 /es/languages/tcl.min.js
sha384-UoNucfdrpjedhP0857F0xFf+0P7O41myVLjJvGk76FMrO9Y5JvB/fyx6MrodO7Pf /es/languages/thrift.js
sha384-ywhHmG9ovsaI1Tafu3/GwgITztNxKBDgXWUX6K3tZx6OdHiyG9dt63QWtjpNIYXY /es/languages/thrift.min.js
sha384-Tb4JtofxkAL1pssPgH+1ibTcndTxx1KcV7zZopkaYBMsNq1R7YXbnRkc/VK0qw07 /es/languages/tp.js
sha384-1aXkr1dOVq5cVBBRu/0PkrcKm0E4T3AGa9M+Z7FZ8xeaeEeyuhZLwm4q2KHBM+1g /es/languages/tp.min.js
sha384-bwdcw6SBKNd1TN+cPh9Q2UYBkIaOMQcCrdFKtCgfxfvmr1m4XgMDwi/euMlP7qqK /es/languages/twig.js
sha384-PEMVkXQ9Mi3tt5Jjr17Xob3ABkIXSzTpFeN4Nl4VXwAWVM9vVFIjBbDzci0pZxFu /es/languages/twig.min.js
sha384-BcyijKQAe0oJGoEBf0y/+dTJjKiy4bIAVdjreJw+MiOkPgCEjM/2FY2+W7K6tcEZ /es/languages/typescript.js
sha384-Mfjt0R07QBpLXhGWeCetLw7s2mTQSoXmcyYnfsSNq4V4YG3FwayBjxod9MxjSB1x /es/languages/typescript.min.js
sha384-F5/TKqRV0bgDO6gtKiVurfUvZH9QrhJnxm7ecQ4tmw5zdP2Yj+zR0uKlOhU8HoE0 /es/languages/vala.js
sha384-nqUhAVmjedbsaoQqA4WSUO/h0PXZ3jktQTj6QCi8D/13poAy2IEhdlydTvJgw+vW /es/languages/vala.min.js
sha384-7ISc+GUb42ykoAX99JagWl40GlfPtedyQL8qN9eUgbdeQcKWYl4iHiDrsFSaLF4Q /es/languages/vbnet.js
sha384-V5qIvBTXBgJ/JFrBZbiDxU5hJxnRsbsU1iLPvYES8a21SdylhibdxjursRkrWFU/ /es/languages/vbnet.min.js
sha384-U9SYaHsYloXAcmenq9AQkIzglefKxTrikQoCirhBiLaNNM5qi6O0atvjICiu10I+ /es/languages/vbscript.js
sha384-C6xz8NJdWRpf3kD8yy3Cr+y8Wo4bcc9KvUK2gOcMDxwR/AlcBvi2mECx7wjStQoK /es/languages/vbscript.min.js
sha384-pu7G6PD4gGJh1kRx2ukGFeBv0K0Z4pXkyu0AcWvxZ/qB5NkRXESMh4t0NHxMtn0I /es/languages/vbscript-html.js
sha384-hX31lddqojtc+QTe5Wx5sLiCvFSEVdP4vg2FEqwbM206W2c3qfhqz+rjC19nxA3k /es/languages/vbscript-html.min.js
sha384-t1qgoHZHiiSTj6BjC5oaU9wYqedDDwqUHrkKztn2/8Kvy+xxVhs58pHnUlFGWIqZ /es/languages/verilog.js
sha384-3cj0wcUVuyPZD7/NmQZVe+R/5hKUXDXlyokh6aPqJBzdN65LnXrrDuR8JJyNrKvN /es/languages/verilog.min.js
sha384-qMVhj1A6BH7PuDigiBaNE2AYrJ4mImzcMqW+ouM5938g+O7uyaP6ygZcs2uA/mr+ /es/languages/vhdl.js
sha384-mdQMmzz/pL52qaQqnXG63Jewnl+lDaXRUeF60ffBbU45RuinqP9EDGlxzyALbe+/ /es/languages/vhdl.min.js
sha384-4601pgEctPyGn7PKZTLv16bd6cWJLZrCg5a+/9ARj1cg9na6NwC+L767YLzdvSdk /es/languages/vim.js
sha384-iGCHxJTxvjzc4eyRff/+05+KaernFwSKiHceFSeDycK+vr6zh9ba2fMppsaMscuO /es/languages/vim.min.js
sha384-TOvJ6ng9+2cEdYIo8UmcklrWBDxDmcyIc8eZ9tntvTcornli0usS+D1YJKmhMdU5 /es/languages/wasm.js
sha384-tu9x6stUNSb4pXeVRKavZtTp8ha5aQOjCp464ykdPSuI1BMHmN/07U/iqeaoKLgE /es/languages/wasm.min.js
sha384-e5uOc66SRxJjfltmoxj6pHJtXEcXx1HC4Q3gHko5erch+hLzR3zZ7WxFQSPKjkwt /es/languages/wren.js
sha384-lPuskV4gmQ7AGr7uwBvKGWgqsw8XALGRb4p75G1Bsy/543YwsCNLejOVJvGQSTci /es/languages/wren.min.js
sha384-BcjZoAx+JfnWI/YaICe1dbsNQwCCBWJKS5uLzyKKMiLTVxBqOIRmaUoiJnYADo0M /es/languages/x86asm.js
sha384-y1FgDrVkuSox/kv/Oib26ZqzunABWyUkirzc4i4sthq8Z7c/ReWp2XA9DCh7MHQh /es/languages/x86asm.min.js
sha384-Yv70iHBx7GsTb1U1ApDgZrEcf0cZCHDHlaQaqCh8Oh0IDfUmS0L2s+q55ezxicvr /es/languages/xl.js
sha384-vHiTR3Tc9YfZRBaKmg0YH47W9B+ElJ6EH5jEEaV+HSoSPOtWe5TdrIHaHqf6uyfo /es/languages/xl.min.js
sha384-Tdx2DY9ZTHx3KhVXYqOVKx3q1zOboDGlTTv8sgMlER8y4WETtqL+C4VQ7B4A0OGq /es/languages/xml.js
sha384-n9ZezaAVj8pK1BIFZQxmC1BM9yGuBNRgvsOxHMHPCXzqYd1gSYIu9KjgGEm8K57w /es/languages/xml.min.js
sha384-7UI8gU2qg1tFMoNOK/cmg7x7ML/l4+FJurYKDa1dafeL1v34UkMPOS56vmEfT1da /es/languages/xquery.js
sha384-4v/r6DeVAX3yqdbicGVQeihQqLgJfIGnc39OTy42Up397bntu9taUUidWN338v10 /es/languages/xquery.min.js
sha384-40MP6/ECSjYaTAIf+/ibE2FPeFPQ53WbASndXxMOcXiQtgLbGXUStZVuPSngp7OD /es/languages/yaml.js
sha384-vXmhozexi2dHQBoniIEbWI5ZqDxyVfUs96BUGpqjWL1aberSp9pyxbvK8WCNASGB /es/languages/yaml.min.js
sha384-QwIudrRGaRjbHWFB3wzgyb3kBeHgVQyAuQXB0KuAOKbA0EihX8lqvJzKWP3CVDEo /es/languages/zephir.js
sha384-aWVzxOWmD+p/ZLEDee7zrLBVPmcuxVaxvXqyi25ezJS887Gk3cWf4Z+PUpU1UUsH /es/languages/zephir.min.js
sha384-s7Qnc+9H7xHt8wvJkvux4UABFXwybyJMRFD+qgfUJ4VLUzd6BLOTikVfIBEYoFJb /languages/1c.js
sha384-yNCP0sdwolnj5W6b+rBC4O9HhTi89S+pMO6g3dhA28E4Nh8U2sODAuKsz6C0BL76 /languages/1c.min.js
sha384-4Gbj8btJIDh6G1JSzgEehxf5UcvWwUM8lvU01XaiuDMGEEnk98BJr9XVgzAj4wxT /languages/abnf.js
sha384-AUNDJl44vrpeLXyYqx/XsT+1pMNhWBAImuI7Vq7AaWEbiAlfsD+c2eeP5MevsDBK /languages/abnf.min.js
sha384-yuc8tauP3EWSVVc7vXrAEw05ILYmK1tZpwl9pOG2D3aTNCij6+EN7iB45aoFjqXY /languages/accesslog.js
sha384-jSRPpx6rrJge30xNPZiRzm9KMbyWxprw/Are7Y34dZ90vGIUfntrU1T0W+a1RhBq /languages/accesslog.min.js
sha384-iZJK0U4Y9iQ93vDLpaqIklBCpLVLk4Jk8zEbaT+cNs5MjhNRSee63JdnnBitU8p2 /languages/actionscript.js
sha384-YSzii8vq7yQVui2TRE60pvJTcXrIfgfUQv5Q3NFkGZKEQoblb4GCykfouqDDPP+x /languages/actionscript.min.js
sha384-PsJRbfffpLCx2TX4kczZf6aOXfqdgn+ncN2LYWXfgu5g0u+ahdF0trlZFjK6DhSu /languages/ada.js
sha384-C/sy2YHBBqSYuLpOIPwmowl8FAib+W+qL+kLPC16NYJzd2mPa2sKsVTUGHVGMNKl /languages/ada.min.js
sha384-GTeEtn+COzT2HqWjaqjmf4YWBFrmlgfpjpAYHE5Psx3vsZf9N16nHgRCx04bpoMh /languages/angelscript.js
sha384-B/SpbqqZVZQIr1t1sf7pgdpGvHqad0b3akYTAoMPuibwOcgvip7stUnu1FDCOToc /languages/angelscript.min.js
sha384-CmG/xb1duBmVNCw0Yr7ww0kOxeBU5qY03kVsiLfTX1HDDVs4NiWAL+mx88jxBtao /languages/apache.js
sha384-iUQ9BcUA+p4w9sZAfo8E9iui0Ay0KwBSTIkTLZB8yc3+ExNWmxWMYMLxB1Q1NOZt /languages/apache.min.js
sha384-yT2h9v4yE2Auvceykn+LhHfjtMdOZRZZ78+POjlKH3ELxv31LG55iDcPPpl1B9D1 /languages/applescript.js
sha384-/ZGnKBtFlWz3dap5TeG4EtewOydrfxdAP22FQI2zXLM9BtAVmutZf6VLqcnNKwoy /languages/applescript.min.js
sha384-jG/quEM4c9MjhAYP7xllz+9G2y0ed3N147caYMN3Hptosdtr+C/38AnJxDnhXP5O /languages/arcade.js
sha384-gGstQAp8sFZHpWlwzuHHRdaRSHf5zRmuKOOxna89kSS1igYH4b1uXlkPq5619I28 /languages/arcade.min.js
sha384-TRdlxCSNmTNG3v+joTZ5Y4I/4Xf5ii/wekG9tuSxcHAf9BX30vv3+jrD0qR9WXKv /languages/arduino.js
sha384-1fjvRLYZVspF/TsOQHIoipbUkoNfSWsc5/FG6aYc1rnvRhLFDZmjuEJRiRZQBgzb /languages/arduino.min.js
sha384-4/eoaXGY0UlWfANb3jyU/u/F701OSu1vFrwgCsXudMnlDIXGAZ4jffvuyiBgg43d /languages/armasm.js
sha384-Xg2bPVtjGJ1EYwh3oKsNkG6p/YNRk1ZGW9IRCbYfFccqH8e7Lm9JPcFZQnMOzNOr /languages/armasm.min.js
sha384-XQWxU5ed7p1JZtCL9asG4KebId1+C4otC8gt2SOjnhgWGZNUlFu30TAYkALGIjLK /languages/asciidoc.js
sha384-q+4aQVHnY1FwKezXFV3xmaRkdKNu1ux80j8yzOSaAjn9EeziuDYQhRFPdqeNXsds /languages/asciidoc.min.js
sha384-9mUDRpJ9q3lSTjxV2M8pQmIg9/5u3bbMedpzzgVzvd/rutrOe72ooBbRqE1Rb9Ri /languages/aspectj.js
sha384-UgvIYJpRQTiRlgz+ApmrQmnucu+VVWocV/co5t/el2inOm9cOQFcIyupaYsoviX9 /languages/aspectj.min.js
sha384-PMBIVYfbEqAVg5xgXN2GSnY/7LAfl4/l04tXeX9Rv3h7rHMh30OdIYpi/FoovCeT /languages/autohotkey.js
sha384-oiof9guf8fns9264wmrURvzAJJ5mv1Ih1dah3NeHJ/IrzCN2d4cA9DObasCXwcA4 /languages/autohotkey.min.js
sha384-wSHgpO9cftCNNxFADLH1yUugOmz+u6PVzveLen3A4WvJfZ2fkf+I/NPbzobjBeJm /languages/autoit.js
sha384-430zfNyjJ94+NW90/wejFObS2O5/ArJ3y1LxoAj8uIFCEjiFW+/lE04XJySkVyKj /languages/autoit.min.js
sha384-qrbnKmQSIR2eFHw3gQ0oAhfxml4kogJL2hYBa0Fw7jgHojIb+Aftx1ytep3i3Qmi /languages/avrasm.js
sha384-nHtU1E068tLBBDCv0xqeAbGJxWj+5WhJyMYtzOdNJI+k+a+2aqW85pZvqINeKhUl /languages/avrasm.min.js
sha384-B7brENEiPAVyc8TUQbr3AR5Omi+knfZM+/EOo3TNSuEm5wa7pAZjccc1QrO8Y1bw /languages/awk.js
sha384-eFizm8+WBkIU2d3kalqH2vV0pDtyQ1xPXQyCROjwwsryMMpzIMQgw0CSz+6AUssT /languages/awk.min.js
sha384-Gh23BewOKqpiAT7qrbcZiUnRVzUXDqJGfCwarz/4uS9yM78LBrlTn0REmiUO2q4J /languages/axapta.js
sha384-I2QKqUbed9Y7Cn3Burg4468VsVX6chc0t/APBZlDAOWzvCA1JupJfLcU1fCzpgKI /languages/axapta.min.js
sha384-4SbTAv3AX2fuPCpSv6HW3p07YgA7hFfcwG2zJHtYv0ATIt1juD3IXj2NSYwTeIpm /languages/bash.js
sha384-83HvQQdGTWqnRLmgm19NjCb1+/awKJGytUX9sm3HJb2ddZ9Fs1Bst2bZogFjt9rr /languages/bash.min.js
sha384-uPOdgF4aFlSo0lFEkjNhjYvOoCpunWFcet8aVcF8gSS8OBqtqk4Lt+Lniiqw2W8y /languages/basic.js
sha384-4PYLazOmBsX2CrXzJcLCBLwOPrvovuNEQptMmXmjugO0spaxi/+Qh4FQrCxZ9FvG /languages/basic.min.js
sha384-rdaLiSrmqCsSM4/sjgeyaVL7ow1Mhns9modvNsMHQdAKPtmuD/mdcJ6hTpbzmF+O /languages/bnf.js
sha384-1rEjTEiJfyW/zOc6+fBt0yfAomOXS+PVFhbrsEchIZn1Og+BC1hZd1ITiibC+mqp /languages/bnf.min.js
sha384-tZTyLhiS+qlG/qEz6Hf8ft/4xgN8k+BBBI68RP7BzZ3UU/kqlxmXyfWZQCMWyDCF /languages/brainfuck.js
sha384-D7Zn/BuvbS4hog8nu/2oCUV7tIcV71MqakBo/xlKsbI76kqn/SkcI9XRYNfnBU5v /languages/brainfuck.min.js
sha384-WHdxyD51Y+ytDdcYGVkKHQOThUwwhLl/1GvZxHTHL4ImI4NS32L/B8bvB/1zN/Mk /languages/c.js
sha384-jtwnwOYA+K4zYN55fA4z4U0PTK5oEp4RcLYaXkYRKO3UUzge1o21ArmvKmTRdh/d /languages/c.min.js
sha384-Phfr9KBsyRX2PtrcHAPDapAG1o6Pq20F4a8oryGqYagujS+OV9PiIbAnPV/zz1Ie /languages/cal.js
sha384-z1DF5ZgOM0rgwam7tgSDoIrODSfOFl4KGa8qxiQTCDUR1JGxqBlQQyS32jAvA8nT /languages/cal.min.js
sha384-5ia1ct5dtxODOf9Cr6gC685LGegl2FU25ztjWcO1RhRMJYev9ndcYq50mrY7gkq6 /languages/capnproto.js
sha384-u6XTarvRv5GvfCSwf1iwzegX6cqsIt+JhRL7+Aq1KnnBOhBPBCn+ysur3NhSl0/m /languages/capnproto.min.js
sha384-xJqLDAFyCKCi4vLE5U06uI2TN70YxSvspflZ6GpcOGi6/OLj6R/j+4WcrHX/7Ovl /languages/ceylon.js
sha384-k42POIunJdHLLeoPI9dqcGAz5bFhJwmxqAiGK+QSK4wXUfPvOBwgBekZQ95jcTcA /languages/ceylon.min.js
sha384-qzu+NU50jYbVQwN8qU9pGXoLOwGFLs0VaiNbOdQoANbr5sJ8tb82C9qtGR/rE+cz /languages/clean.js
sha384-E041Cil/vgxM9X9fEtcmpBtC9xKD65kaUVTyBinVmPhk/EU8FcKUaKzTaMeW+yrV /languages/clean.min.js
sha384-asfBmEo7hkA2rhFAfj7n84sizA0I2zMcSKD+ldR9+267g46n6z4pPa7xaBcwrz5E /languages/clojure.js
sha384-Wzcbu1NwxLbZ7syk2AG7Uel/YM0wnO+k1hZgcKkqX+Rtbc8ezBkEZTVwsSu6T1M8 /languages/clojure.min.js
sha384-Vs3C7Ach4fAj4HKU/t85K+HwTRhh8qn1T33/PtjdqZTU5ylf6G5fRm/yL6sBZoF3 /languages/clojure-repl.js
sha384-xvji4qjk1yTch1tZsD6rflf99DB1cFqcQ6OeO1nwb/JXQ6/6y6pKzgumMTW7yrCQ /languages/clojure-repl.min.js
sha384-gXeNvrs51WLYPI/WV54m5BSGtmPSOkEngcs0hrofwSIZ+uVCXW6SwnU75AWUpsiH /languages/cmake.js
sha384-NXUiKRE4iB8J7h3t1Z2aLjDRJslCWi4SWqJpec1z0Y927kNqWejg1uGaEh1P3GCK /languages/cmake.min.js
sha384-NvnoZqkahHVCl3GCk1wFvZ2XGJapETINvhxhdR/EyJQEXkwjYL+zsidyoeRIRWsj /languages/coffeescript.js
sha384-j8a5RCWRIwrFdX0knM7cz9bqtHX4osYmTT2CBpXf/rlQfiSKq6cRbCsxLcJWPoxe /languages/coffeescript.min.js
sha384-t1MYkz2Fb8FyrxlwHPWWt7fRGpDDcPLHuMuYKn6rjvWtLf6lYMMVZdlmJFoUjQOa /languages/coq.js
sha384-LFxbMDVPRH4AeQG7k/EHmUwU0yT7Pl+q4JL7xqBnZXy/v8HY+RNAssnYJKzbaJ6t /languages/coq.min.js
sha384-bcWzuYjSS9NukL8640ddAflslUOSjMO5/odjI+kRe0n/w9sVbmQjASPzHiBiGun0 /languages/cos.js
sha384-pMjYEVk6bW/Li5fVrTdpSGd5dGEGD/jrgHSU08VrcmsqyZB2B/8245L9/m9F4r0D /languages/cos.min.js
sha384-M2wpTxQe2N0750xYZ0zTinwbmjsZjdtuS7twUUP2dxtHR0YqhY3JuUFyyhANf9Uy /languages/cpp.js
sha384-/yf54L01PbO6NtVs1Pu9rgfNHbKXanLdNcGVuNa0m5+KiyH+1NpZRDK6idm5VoVl /languages/cpp.min.js
sha384-oVRnH48PG/8qn83d70695zzTZBijDFSJ+b7V7cUczSjp0EDP6o8j7mTLzvhRJOLs /languages/crmsh.js
sha384-pWWH+XECnKlN3vERExCAowNTyRmNartFm9vBbd9NCeMhtyAFqHE/EQLE1fpGlS3c /languages/crmsh.min.js
sha384-Qe84jwg61rMxZr+KUcb8bV8Z3EWINxiLU4VBXoNqAWWgkxvMFGixcGtTytcMGpMU /languages/crystal.js
sha384-vsyuH3iggrwNYvo/3BmSS7DJq9olpDb+P2wlR7ziu2vDbXfyGGzE+V+3BYVUoJTo /languages/crystal.min.js
sha384-73x+NDGuWTdFik2BOd5uwmBcikSmR+Qx5AVbJLifM/M0hBbwKToQ45xBWxKYkpgd /languages/csharp.js
sha384-6NsOlZtv7W2iSoDU+Yi+hENfl3MuiECvnl7emdRUvpIpDbLvoCjpAU1fjE6HxIQp /languages/csharp.min.js
sha384-gf4BcDpEnRsigTWjrncudvt6aXYYJuPqHO28JCgKNG5BtKKPR3YIU7M4X77LG/hG /languages/csp.js
sha384-GfsA+L6x4v9I1iiHCTfElgEMXs4tsXCw+gC+bC89n7nSXfUEfNemTt48EJD4jndY /languages/csp.min.js
sha384-h6xPJgkyvp13tIs697wZHjCH20tW1aJOrvnAKiZZiATSWZp0lyLB4bAdsEhWUSze /languages/css.js
sha384-+MO3D3y/aZzZq7QMAAA5KiuAcqBZivJHFmVUXfwdBoLxEXeGTeQGsNMll4fpnegg /languages/css.min.js
sha384-RnMFeFqP6RHJ0xU/nSvD3wnyQxqDOIQIa/ZZO+M66stbWROhzxp0E1TDnyDM9U2t /languages/d.js
sha384-H5nvRzsubVRK0OxJ9XKWgoOkkYJTKWWthvtIgsZIWYkopc102uu8/yEaIr6Jv8lj /languages/d.min.js
sha384-dMNLrPkwjJLQU3QnOACuNxKwF5uJKlgc1jxuGPSZgOTAhFFsakEk5maskn1jbxjY /languages/dart.js
sha384-xV+yQq6yNCJpHU1bXc40ygKBQS6BF2u/5RgLa0A64Rn/ue1fW52gr1gF8rko0wkG /languages/dart.min.js
sha384-OYIpfrwVpwa945Xtvjo7YUOg5IXXXP8VeOvrGtPaLE0aILTQkcBEywDnV6DABQzt /languages/delphi.js
sha384-WPXcFtkOLxX0vKdl+zKh+zWbdMF2GV5aWrNbGT4L2qBYLSUxq1KQDWYhJHh8qRrt /languages/delphi.min.js
sha384-ptbaKMqucgUUAhyaQfodHtSDpTA0AAoyGZZqos59ECIdi1qKRnUZcLOxMkZaxkul /languages/diff.js
sha384-IZ99gU0P2i3O8itOlz4exVdl6lGFAgj7zq4hgyoe29bt1KyJykBSCxdH8ubn3DSk /languages/diff.min.js
sha384-gfyiMmAn2+1KPBU6zgf5/GuxwAS1MG2sXO2QNIa1b8gWqS+ZAEGASFZUv2DTTr0I /languages/django.js
sha384-E8a05vTFeTlJrGsYy6uvHSN7YyYtYVDfuTB0NYKryrTqkKAh0Kljndtou/5x9KxE /languages/django.min.js
sha384-GEtDae05TD65Rudc09ryy9h0sXTp18D4g24uImkYj4cC4pmv9QSL9gpFFduB6mRv /languages/dns.js
sha384-KEzcnJSlyQjvNpDwNeh6LUBpvcvNSFDj5NMNxUiH/1XzSiutw7KXKxsqF+LJm6+P /languages/dns.min.js
sha384-KgZWfCUcAWOSNTSNOBUrbGToPbSNE30TSimcL9oKDQ35EApOQoCYU4o2ayix5Ohe /languages/dockerfile.js
sha384-jg4vR4ePpACdBVLAe+31BrI3MW4sfv1AS62HlXRXmQWk2q98yJqKR5VxHzuABw8X /languages/dockerfile.min.js
sha384-2AGuNVqh/JeqLUsnaXugKMfd6DQa+kdqeeWrEhLTVidJ1i7lTDFop91fqgy9Y7fL /languages/dos.js
sha384-5NYiSl0YAfUusQHfYmh7TNKO2XTd4alWFiwdqemWAjv7qIAJt3Z6vm8J7XDj+8dY /languages/dos.min.js
sha384-gZJMvXhgCfmHi4svPuKIYpT8ec8vl6sGvSENrpXsuwpsZ8fgB0lfT5byWlVL+ZEV /languages/dsconfig.js
sha384-UOr3RVijENphF7cBcCuiwwXBTABQ1c2v6GGYxV/cBVDjPU5Ikmw4dIYJXc51cSsu /languages/dsconfig.min.js
sha384-YzoJvIYJ38h5w45GMaZBbINa7kKtGUKX4gx9EKCYvXTx7R+4dMBuJdps/2ZwGTSD /languages/dts.js
sha384-KmsHM96M8R4zfTswDLJs69vAc2OJUogOt8Cm4o8l40GllTIg9C9eZk5Nh0JlKvDI /languages/dts.min.js
sha384-jW5wYyB1JRMwQN+nnhg6mFTc68o3VaAel3q/DaK+PtEFf1IerlzTM8DEZ5yfdO01 /languages/dust.js
sha384-yF7iAW+5SyekbVk9JwdcD9FvBYGbwXM9+uNjWDiZwXY3K3sQIweb00ODpskdmvoG /languages/dust.min.js
sha384-ynESHsdD96DPMXScqdCoF/CqHwtFNNkdhAp8qVJTbolJG+lnz4X0F6JUDyix3iIF /languages/ebnf.js
sha384-zrN3w76G13sFPJ4BPRDmYrPe1TW4P+1wpgAcXcj9J9VCpDkGj+H3HbK3xuQvnXfq /languages/ebnf.min.js
sha384-qtlgQhMe3dPictT/p8u+Re+71kMSr/Zgoe0GwOS80c0tbv1ZnDlQ9E8RQMoPvk/W /languages/elixir.js
sha384-1yAJ5qBg8K80+przprGQnQPGeFIb3tKmAVjHnSSEZiq4HKjh3cNy/nwzJ7l2nFK+ /languages/elixir.min.js
sha384-XDYctiP3nM+f8VZWhnvZyJJCKB0qPSh/nSJLqLsrAoZ7yG7rInwJCHkirrfD2nUN /languages/elm.js
sha384-KGAFeqpaGSoOdl/gNDQIpZxHTyRKmSxbQrHZ6amSeeDxFyRUffqIt+10aCHUUXF3 /languages/elm.min.js
sha384-X3X5mf91CrWWGZ9b3+VLi7Qm5aWxDv2NQDRoVls8Mp5weWmncrpxblLBZ4dNp0QH /languages/erb.js
sha384-gEX03lFlUpoA8rfkmZ2UaKi2820eK1l0gtckxIGjq6Czr2uCyyEfekMrSoFwej+C /languages/erb.min.js
sha384-SlKWu3SvV7MLl/uYjSGXVo0n5bflYsgkvmH8lmKx1XmwA3jKehvoWmuCHcXsMbjj /languages/erlang.js
sha384-0CP4DzMXyug19cwGTs3ONRRpUBBFTa4O5Hv9YjyJY10yvQOl5W+x5LOifval6ZBt /languages/erlang.min.js
sha384-5p6je/gKFM1nHFcEZz+30/mQBcCs9uhZWG4CnGvaucHJ5oCsM9zaN+mFGpyMvzM4 /languages/erlang-repl.js
sha384-/FY+XRWmaEh7r6TTcq3KEOxwB6sp4MEUUh0Ls2PNyfch1P2kpYlIm6HF9INVd7MS /languages/erlang-repl.min.js
sha384-ZSZDNiVxP+4Q4FlDjBkKXq39vsfluRPm3E3REXxtNNsHe3HrDYn2Abs8q3jJtAp7 /languages/excel.js
sha384-cE562Xij2Xk4SxySz0GKZX6CH4pFe4wSgZgEyKEzTYRhiORoEnlvzFeW/2ZINZ6z /languages/excel.min.js
sha384-C3/NNfGAKCQQ9bJFwbtUcZHfHDlCcEq+Pd3L72Br3GfKc+p1CPdrcbNJOWuE+Tn3 /languages/fix.js
sha384-971pUduvIswF8i7V63S9oUoSCed6WGwmHM3Imgy/209KXdtphB1Jh/85+fJPI3TA /languages/fix.min.js
sha384-NLcn6fD9rmfskmx/cQfGIslOo3F2kTnGMMU72nbRDFOvVjWh0HtdoO56eKflraZU /languages/flix.js
sha384-Qz1f08u7z8GyotqYiyJEc0CSlFIt0sbnl9z0bRmcKh1K8Tj7EJXwJtBxpwdTvogd /languages/flix.min.js
sha384-Zc9c8EKPb4/RY6vnZtk/dIUsr3hLRfKyZQYUoYvVCIWuTBJLDjkKoch93fmXrTFr /languages/fortran.js
sha384-4DbMwCqRqcxBersl7bPX3KEVstVtQ9CgGKwlo79Iu61BRrE3O0uHhhoXhuDriGZn /languages/fortran.min.js
sha384-3yn52MUDmkYUMfQve8G9PIMJb+OT2CCM97H3VinYelUgdRItYf1DfkxtQ8vbMHS+ /languages/fsharp.js
sha384-nYNTSA4nfVQ5esuMpVP111sGLunWJMnBXsYufX4ExWifybw5u48EAUxr+SsgIowR /languages/fsharp.min.js
sha384-8+DMclzJf6BdhUx8AR3VekTQ8Jp1YLbwQ7ychU4Z3YPnc9l0WfE+iqtNUjmVvIoC /languages/gams.js
sha384-zwiFuwrESVX82ZurqEjUprf78YrmlWWXZTA7XQy4wCSw8liRMvTT+lzEeI0t7gBa /languages/gams.min.js
sha384-1CTF9Y/daROUkh6yI9u3/o45HcswxnvNRCqduP0h5S1k1DiU5VH4jvbnQf+sLxS0 /languages/gauss.js
sha384-ZXXsPS+ORWfAch+VsKTpwqC2ZAO1Ih6Uqk+5I8Si9yAO1UXLNuK+CgkbSGly2GQA /languages/gauss.min.js
sha384-CBopYXLnw9C7mMD02aXs9U3AdE8cA9xfhEZyH3lqQIo30mjHyse0C5cpALPHYRtS /languages/gcode.js
sha384-YLuaMuZSDa3eAMAZ6uLgoNUFGWVjLLQy8eJ4C9qhla3Yjx6hlGwiUXrs8jq8Lf1a /languages/gcode.min.js
sha384-OMBpbmq5yVKOhbnII28QO2+mcuXn02yWg0XaYqYuw3OLeaVVrObx/PKKWgmpxUfk /languages/gherkin.js
sha384-ppssxHN00pV6RryFPHORyMZIe5RpB6yrukW7w4JdFAIrCpJVJCVTC3pR9KEjDpAS /languages/gherkin.min.js
sha384-FmsaZHRekzHoNL++5S8HZFIJNK42khOgzT3VRksnYVHAURRu3hL5xMgenu9lHyAr /languages/glsl.js
sha384-ubPsF/0wBxL7zEYtpuIL27F0zYtzIZknk+3GdMEcQw1v0Td6xAJ6gDhxHy0UFhk+ /languages/glsl.min.js
sha384-/7m+VgcIeNK3Ah3dxPpfbTCrscYoGuB8+kOOhxwm7ZjtaBZqsER4CnOHUbVZRX5s /languages/gml.js
sha384-FApAVPv3mT368h+qYeULjOQHEoMZXmF2+3ULgzAtrpjRJHxFjAn+Xq2S5TQXrZBd /languages/gml.min.js
sha384-B9Y0sXbhPrwdlpzfeFn4NkyJrhYEUFUCTMrEVRu+d2/3aJ/4ZOjFPJRZFnJdaQJm /languages/go.js
sha384-Mtb4EH3R9NMDME1sPQALOYR8KGqwrXAtmc6XGxDd0XaXB23irPKsuET0JjZt5utI /languages/go.min.js
sha384-ehNukSFKM3Lu5zZlAC31w+OJ/VeSPgzCw1S829YkOdvmhv1gMYRtOVvbCI6Nbk/7 /languages/golo.js
sha384-VaLJgrWWfCVtbAeOjBIOf/CbAIqj1IbdKtJhHh+gh+2H8F7XEhfK/a5dcg/lhxXH /languages/golo.min.js
sha384-IZOqI6gvg7a5fn8xvCBNYqwORJwHrhNcuB0hu3O+gRI7Rd7VPHYaWkvSjz+ve/F2 /languages/gradle.js
sha384-Qh5sDx8XR/gARjQgTGqZrWsG9IWURWytCVZFqqUDtx4M16przYikOSpBofp/smEj /languages/gradle.min.js
sha384-y4jAHac6QZpx9l3FE/Q9CbTHzEhF8S7+9Hm8AY4PfpSCmQWpPCdW1rBh63nXebrT /languages/graphql.js
sha384-2vKrGN0+i0vPRkMvpEZkOgi35/Vyn5pGG9Oi+baBPODVLjCrGyhQpwDG+BBEEeA3 /languages/graphql.min.js
sha384-Fr35sXYVlEaymRuB/DgWmIYMdnG/QdVbNBwYzaLTt6PB40JjymBdX1RkmpxAjsy0 /languages/groovy.js
sha384-Zsb2SvyyyZPjojn07qmkDNfdiapkw/4t6kb2Abbh14QEJBHNsXBhkE9wWfbVPYOW /languages/groovy.min.js
sha384-5SNMIKu96tFhXcQ+lbTnyJN+IAJrnl6zsOQHlIAlMnKnyMRPcAkTNapBnGzfkJDw /languages/haml.js
sha384-uQS8ZRxc1TFIVag12uukmzjpuxnV7ATxgtJ9AQ4V2bcFtNVQK0mpSh3oIjLcvipv /languages/haml.min.js
sha384-nvXZfYfyJ3ss6utjLlGTuEcrLU7KoKUzFqjYH+rPWK5/WvEJPPkLd/AQY7CDJQL0 /languages/handlebars.js
sha384-B7PNh5MCEV6Mz/dVdekDqHupFqAlPHt9qknPa+BHdOlUd95jeTW7QByrAalohboI /languages/handlebars.min.js
sha384-fAOcr1krCNXboMwn/cAybBtSHzCLUrlvVfJT4decCiteyOsqee0RnqLS23UiUBNI /languages/haskell.js
sha384-1+L6SezuX8EOUBfYZyp1GMjjPHwIiJhMhRsLebXPt5ItYffwrwQII8OW4Zu/hndV /languages/haskell.min.js
sha384-p9rSfF/IKMPASXtTFmiI0PMuTZQyy3gnk3DkppFB1FkOkGlGczGa6vfhxhvAwWdO /languages/haxe.js
sha384-BvW6WsJimyvFXi5xeOmolLeUQ+pMLER6sQzMGgq+6aOstIyN7go9WEdmE19Vbyry /languages/haxe.min.js
sha384-Nax8pKtVD8rmv1lPQlNia4i91TtJVE0LfejzjRPu8VfB9yzss+9xq60Ab4OPWGQZ /languages/hsp.js
sha384-AdGtHB9BrPseyh43FlhtWRXQDcI2rRahPtIB3Dp2XBvhE+SB4kPfMyh2G8HSHXmy /languages/hsp.min.js
sha384-5njNAV6cayF+v1sc70/t3BTkztvcp8TZ61d65U8YUQuXJ45PIrhcgNfccRMd9JsI /languages/http.js
sha384-ubRntct0s40ZDtDRLkxA3/xYX51o5yC2U8SKlky8dhIRsjSnvZiUKLhz0gNTewno /languages/http.min.js
sha384-CPu9pXWXcT3rC5TEgpg7hryzWwxOsxHLArIS+0JUXzVEPglprQiXAMDWKS+XmcsK /languages/hy.js
sha384-ZaQba+1+217wybfm4ZhQcYeyF3wdFyT+AeWx9tv3saciWITHU2rWUhYSYxunVb0r /languages/hy.min.js
sha384-sjGC2mE1JySbV1HpsbpO4/l5KecDvcMwrmkibtJxlwOGILWsauL6cykvpg25inSc /languages/inform7.js
sha384-eQgUZRyLniTo0FTkATb327W+4GXuOZwxmBM8T9lQDg6JtwltIb6Kvv+eNe8RsCFy /languages/inform7.min.js
sha384-l2Aa/1StxIePW3t8ALFDwO/VZShzdfn5Y+0qIFkvO4WXem4DA5+6fgKQW+w/xKEk /languages/ini.js
sha384-0/1VV9gfjl+ZuUf+R7fvp6dQlJ5JVh+WzEqjzOwd+PCh8fa104Vm13MBaJjTz+cG /languages/ini.min.js
sha384-MMTttJH73VYW+POZuDCaMs1/S0yPgmXRsye7CmDPeaBx/CXGJJJB25kkoHthBzQ/ /languages/irpf90.js
sha384-nOpxTlWRgLR13aZWWqxVWHxB0hn+sCSwnTWCcC2TNJISGlmzyU9A/R1Rm/0/GUw9 /languages/irpf90.min.js
sha384-eoUZplJRyJQNUFWSU6swy7HgwlhBWGXJ5Nj7pHJH3XWnQ7lfx6PgXxBOrsJuk00j /languages/isbl.js
sha384-nAq7dvjNYEoyfKcI9TVNM4NmxAF68eGURBz/2PtasIfUkEnkb5POJ7vzb5feIlUI /languages/isbl.min.js
sha384-cZ2d3Mo/jmTF9r2kHWcHmA8hehxX8N44UN6LSkEhaCRe6t8e9ntd5JEuafywm0aw /languages/java.js
sha384-8mc5ynnm3AlnXn8P3ccSqVAaZIDoijPM08/Hp4DABy6GMy7EHCQFwiIUoGAaGJiO /languages/java.min.js
sha384-p/utwvqrRVOLlz0BjJ0BCGCb2liTDipfz47/QmGXz9hoPIjCKYEgmYUC30VmGgZy /languages/javascript.js
sha384-L/XmDiyusXomLRGcRmcBpPlboRFjpQNV747OJvg+sEOpgGYvUsNwcC4JLNQ2dI6O /languages/javascript.min.js
sha384-99iaXkQLPR3ySoGmARWG2gBaQM9b7Lp3z3j1G43bb23L//VcwFu+iKqLyy6cGsrm /languages/jboss-cli.js
sha384-lUROqArhKl6lBtLfj7c/6cvdo9MeijqmAlpclz5Qr0EvK8ttOBHJQ7CZPoINACyM /languages/jboss-cli.min.js
sha384-psmmPlbfEWGyvRapexDqkVTgNz7Y1xvlGdLNWQSafI4GFQYFDXPZxVXH1laU4n6l /languages/json.js
sha384-Bb6DhE3tUpBROwypL78TbhRUs9QbCt2GxcxVSYglt2l3MefrYkm4CfwjfWhRfQaX /languages/json.min.js
sha384-sQxfxDX5OQafNaqMp/kp6YIlubAAM8WwqsJ1t1sn/bHXq2G6Vt031NBDZU++xioW /languages/julia.js
sha384-aXaL309hpQjGj0iiC05w9SImeyJiZ/9XKOnzrg9IiBEj5iaO2UTe3Or/KJHfOKFc /languages/julia.min.js
sha384-oZ4dHlDLvKfEEPFi1fvJwonxGPjzFwQBZKujUFqBrgzEixahGOvRUn0YVHFYlfdm /languages/julia-repl.js
sha384-sjuRI7LKtdlJaa/cLtQZMJIJKg2ZLleJSmPtcXZxe3FxEhBMktIkyiQTpkP7byoq /languages/julia-repl.min.js
sha384-Ve7wqoYcjaWimhzLnfK0sj2Lij8DmxK2diJ9kazkpifUrd7O+b2CbnHSxD3SCzqj /languages/kotlin.js
sha384-s0inyAR7LE5SVvn9VCZrQaiUxkDi+RsQdSKzFh2CjWf+LFd01DAjt9wtxGhT+4qT /languages/kotlin.min.js
sha384-aNmKczzJD13MpG2aKYbenAC+a+jA0fxshAjIYNe8t4XIMS45jywxS3icPET1k0jt /languages/lasso.js
sha384-ZZRcOijvFcSYDPwKpgTEOGSm+Ww/Dmj2lwxelFyQsl4U7MHHlYDt5vEyCdo82FhG /languages/lasso.min.js
sha384-dllA+pkm+v2eTaDGPSVPCVVy5zE8kkFNxjApnQCXfjf639vbck+YVUTbm5UIlTcr /languages/latex.js
sha384-YnQI+TwzZ1xRk2HJrBDWxf28lhGoOKeLoiMTznB2mX2JDclqH/iuPsQC86Qtwh6X /languages/latex.min.js
sha384-Xl1G2r7SYKb1Wz3Q/KWsNCklPT0Y3gPLzHi3F7ke4DJQq0tWieWfO2jA5YIuAwTk /languages/ldif.js
sha384-sJB50VGTeIqTEHDgjZSgd1y1E7nJz0ZY8+7RXplspljiErNkS2/0c7/EY85jhIs4 /languages/ldif.min.js
sha384-gguYkq0KYi453IxkmDF3CbsqtFZwRw812Hoy9pijruZBUVVQzJw8MbWZJa+0G/f5 /languages/leaf.js
sha384-4g0nL2h6+jF1QzsdR/uAFRuswDrXVI5SQaKq0Ig06T820naSMaIFWtWb9gs4/RB+ /languages/leaf.min.js
sha384-zlcx7qQtpp4cqCCPEDhWKvb3lEbhomD5HTubdLR78S/3AQBJGkHiT6LGziV0f4yH /languages/less.js
sha384-NZS9QHMmwN86GbNpIvGs4HOB2EzfqPjhIHqZviwZIw1HLumSMRnl4X/OIgpyP0+K /languages/less.min.js
sha384-wNSbLbrz+VIFN3IZVDaMcTApJCooEC4uORZoSfvEbg3cK3hTpnZXhSUlD8cbrYuz /languages/lisp.js
sha384-Rt/SKBZQ3kgIiK9pZeM7wlw3VIC9GqKB5CWElZkRBdmqW8LgAQyzMIxNaTw0M2j3 /languages/lisp.min.js
sha384-3dOUfU5ms0kQIIgqaY9XHXb+xLtb4ZLRk7YEDZ1HQDolpkd0t4MOAbeCKv1KNzTC /languages/livecodeserver.js
sha384-bVyarbf6VCg82623c+ljGhsOzwmgyE177jfM0o28keFDzQfsv9ytARIaag/I6100 /languages/livecodeserver.min.js
sha384-pmtJBf/XXjMmhR9aRBWc0sHa0VxfwqlYyhnkRnL1T1MgXUZTVFD+JOCD2sm3Ry8x /languages/livescript.js
sha384-8yK16EmWIAHNcs1eB22u5N46+eXXbOw5Ca13l7ExJmgEw13ecvKn4qQ3Vri/neZR /languages/livescript.min.js
sha384-CAesSA8LQoXhLarhdcoHgorC4QmHI9t5Jn+0n9OWH7aNpeTuQTdsLz4jpR02c1mM /languages/llvm.js
sha384-2E2Olv5Np0C9fY2pt3A6kvYCcGDTyGzMc1HxkVA5d0gczQAz7YiPND12qr1xDfzG /languages/llvm.min.js
sha384-tSKu98ACuQqhbsihSBFTS+xFDb8m1B2MHlI7e7Tesf7e2OgWv95R9xj5TtzSj13I /languages/lsl.js
sha384-uVH2rCvbWFMkwVPsVokIzvS1BHV1h/0W8K4Tkdo705C4Zsdz240UI5llM735JDmb /languages/lsl.min.js
sha384-N16TmpAW2Qx3T6s3/C0f0jweUUSh69YsJ1+AnOsAePpjmmydXF/mjeY0MZEF/cPn /languages/lua.js
sha384-WES4Ky9nlehN2gBM6oY3/98hVtWF5PZnBO+7jK8ZVKFo+QR32zhfuRK+Mv9jQsOK /languages/lua.min.js
sha384-Z4QQuz3ChYj9P02v2CDc+Y0OAn3iWXtJnyNd0Q6QqW4GV28viT3zcS9tYSmb9x1L /languages/makefile.js
sha384-pYbMiHWycMKEfJaSEsquFRDTjCY7QHvQN0FIfDK0lVMd9DPJuOA7Kq5wZGecvYwM /languages/makefile.min.js
sha384-TVvUXbmPgdS59xZSFUeyNQ1vUkeCbBpuMj3qlzdEwdQhoO5F/WNdI94UEw8g7Enp /languages/markdown.js
sha384-sFh+6oaRBb6kdaMLf8x7qeH7NTvm2u1Ta6PtI0S8hoA+bP7UtHCyKFzaI1JBXwhT /languages/markdown.min.js
sha384-cv2WQeu41sqIpfVh/UMjGK1yFTFZ4wCxaGbFIzi0+KvJG6lHtSQYcJccQvHu14iR /languages/mathematica.js
sha384-1WyRNxnxiRASoImr/x5teZlt2xlwOotnfkJAj33imuGcVgndwsuCeyHAIke2RKmN /languages/mathematica.min.js
sha384-dVsmHbA9CPX9J4KJRy5BJ3jh7TpUIC0/P2laA1MHxXUMMUboT/umHOdxyekoDxs4 /languages/matlab.js
sha384-SEUclvWzbtiTGabHIlauQmWVPE/JkSVlJe5BKAnSErPApBGiH4/5zu8+yWtPCUqN /languages/matlab.min.js
sha384-ypP/fJZukFEYC+4cozsRPFubnRiEK0KB3FIV2OY4ABgNRbZJs1EqJ8sEqIvgfADS /languages/maxima.js
sha384-F9NP072dYWBOnlVFrv0gpeItWR7egIQiE8rXgvnYEUZLZJ+J8jmb7gcaEp+UaUk0 /languages/maxima.min.js
sha384-2YmBKpUO4kz3w014lPthisoqAmgxSKhEYqgP3VBApxkj+XudCSsdN3h8555jmERy /languages/mel.js
sha384-dlO3LB/5yjSdavmxp9TaKZblcqj5zaTJRDKgojwiVYPxMdmAUtAzPQsTW/0haJZJ /languages/mel.min.js
sha384-o5GVWDLdq5OzwXoXkm1CA30MytTESy1MdMgZRsu5DHQpnS5DdzsJW5c+M+TYS0/O /languages/mercury.js
sha384-5KB5j2LLN83tz772MpfqgmaHbgNJoGpqncjtBcrCE1c0l0AKBBRmHt3STu3jYxoF /languages/mercury.min.js
sha384-Ts2JYqrveTDUYgCpQAOM+PD73GbTlU1aBGZj5JaY6pF+XUqq2MOZ6w532HEl3pwt /languages/mipsasm.js
sha384-Hc5qAKAeMWj7W95V6BNPzRqhhA2leur3hyGlhd4ncVW5kO+LReHa1M+XfT9qrRC5 /languages/mipsasm.min.js
sha384-BEod7E5ZzhIAuY6yUQBC1amu7HUah9PNcsYQLo6EG/+yrMYuBU8Ul8Fg+EaUm0Xh /languages/mizar.js
sha384-RVJmhZMr2jBnaa18Qeu4zmda0XwGYYPv0C17h+6NlgD5LZMPKn4tJRQuVPxhOvFu /languages/mizar.min.js
sha384-kiNTmJ7sUzNa0fWdrmrQG6CDla+qKanV1sWUwTrektTbSt0RYzgGuaC2CtrpLrnp /languages/mojolicious.js
sha384-iq3FlJUEnFxl1esoj6R7yvUpe4xwB8SRMskdl49eoywRd5vxOTDuiVdb4L+X3LyM /languages/mojolicious.min.js
sha384-luv5o3orI1R3S6xitP7iUuOVlsWvrOMh8Bd4y3q18aVbd7Zv8j9ixw0PtBqcFrRu /languages/monkey.js
sha384-Sc8ePJIQQwgAs0e2t3ISMAssMi6OXVyCfhyUsf+YDfFAtc++oz1hnwPydbojFvXI /languages/monkey.min.js
sha384-mFJPlYKF2qH7R9UqIV0mOF/ROc7vI5M0fsQu5EUTjOnzgnlZbYHVvEWW+ktpXrTD /languages/moonscript.js
sha384-nPXR87ewAxssB/YfzBwp1kfvUtkNP/qe/feDB0JUoBzlbkVbI14vOyEp/BCEUYm6 /languages/moonscript.min.js
sha384-2qj7jabHv/5aZa7xWqnWyR6wSyYbTRuK2ddnnsz/0CZ4DX3uTkn4ahgebsufFIv0 /languages/n1ql.js
sha384-lkHkblFGkRtnpUfwjwgwfsqu/ugnms7NU6Ao+EhElIIPVTyAKANHf9HxMyxkqYHf /languages/n1ql.min.js
sha384-rMekh7lvPyG3yKE9hj9ngJt3n+qVk/owO7Jf8Y3dMPPOtY9iOdJwhQsw1lJm5uDl /languages/nestedtext.js
sha384-LuyPddXj3XboLoEpfIjtRhUGLKeTPi+/8SBIKEm+sYpG/IQexvq3zIAS86PRP9hM /languages/nestedtext.min.js
sha384-GqxuhQ5X9X3c8nNswtucj7gX9fAuYCtI73NbFLXAYNqX39+zocekxv7SOK6oVGhi /languages/nginx.js
sha384-TG8jUbt29ktiHxVaCkA6FLnJkL/PYG3zQwEYexdbr+Z6mMkFf+c0ONpHyuIY4vvG /languages/nginx.min.js
sha384-jSVJDw177bdlPF0wKwdpmQYvJtYk6O8io+8qko+0iCOh+iIhJt7cp6dkrc9/imKv /languages/nim.js
sha384-exQrQUL2NBuy5l35FwAYS9jFajBnrRCmhIM9yrQgdWOoM4s6aIzqW9gkfn+LaC4f /languages/nim.min.js
sha384-e+8SNRLB6VvDW6MD6Lk9ABdku6f5fkp7yvs9mpzbNRz7ad83pNxbQDH41SDYdevy /languages/nix.js
sha384-TQ3GD74bUmAbxGCTp6y09Ujxio1hxtA79eu7/gYKwgFanLtczRqmlf81EJQmY8gH /languages/nix.min.js
sha384-VIQPm0oHkRJhkG6rBz9kCa4E8J3J1bKONvg/bBUhu6UbqcMcBBkbTz95iyb4A5m6 /languages/node-repl.js
sha384-aZXuOYsqBQ2RFl2pfS/f5zLA9yUjgNr0/zeuus2h55Mp6eWqQapLIIJYDsG14A89 /languages/node-repl.min.js
sha384-5eCdMGRxxe8HdiXANMhFTitOaXAivufwoae2yRQ9tC/QAP09y+pE7oQFwADePbg7 /languages/nsis.js
sha384-4a8Q6mmzSuhT+l+DrhqYu4KlkhOyLJiF16gkWqVqjXgPSaK1sO1W5YV20xOoNg43 /languages/nsis.min.js
sha384-eZQaDnoyMq8nBrMePlILtztnOQzUcfoQmwCDCkCRCqAOwtowxxXuSbXx3QCyDMWi /languages/objectivec.js
sha384-evxKJTmk5AChNAzYM6uh+DVNAm7QCNu2x/OynLEpbH+h/TVMellCIci2K+uvXWID /languages/objectivec.min.js
sha384-dRCbGDspfGuNKCQw2l0ng9m7DLitN+3IsbQo4rsb2vSS9URhm2G3U2h1E6knjfyi /languages/ocaml.js
sha384-aFCHrYMEyAT2T8i9tOfudmHsGqppJfwRKg8n8hbAMVPV9NfdCCtdw9DFBWd/xr8O /languages/ocaml.min.js
sha384-Ypi2k+AarbUJzBgygOSgbCBTJnQtwBAduzSqVhWI2CCF3sCCfzdZsYfsD9ud82sX /languages/openscad.js
sha384-Dhxsr9ocIQrnHlppkSen7Dge0kK5KC2XBDPG2PVCDVQdHcem7THu/X8ed9RYhZ0g /languages/openscad.min.js
sha384-DTvc9HJlhtRK00m1OxZhWf1QrLyuSeumlh0ClE3CnzFy19m2RsQGKYFEYmY/6EL5 /languages/oxygene.js
sha384-JVC+S3b8TEbM5xgapLjbhcnLc7YbakzETqPtT3xLLvnm6etYjVjsB0uDPMRkmagW /languages/oxygene.min.js
sha384-mpp8yrmj/LTmAFkMfJVZ0aCftZXtTdfPsBFcF3x3z4ykj/bPyqgHl9I72PfUpXRz /languages/parser3.js
sha384-SIQiLYyI1nDqsAsAbw/Jws1FZPmWmAyhR1eu3MCDZ3NahqRoelEK14C6q6j1Kign /languages/parser3.min.js
sha384-zERbDkBWHytM3CXyujBAd5bpdMZDupVU6hl9bDiSg4w9I8bV6KhGSKxsCdcPWWU0 /languages/perl.js
sha384-HBc3JQgC+i/l43bOuIE9xtQz9ZFXZDEjPCyiFD7O5Wauvl79zHEQmV4uDStGEQLu /languages/perl.min.js
sha384-LByoAlB9xcxZ5C5YRubnccCjpQCEzOaPqZ1+l5R8OmTBMN2nF7g4Afw8UU1XlF7r /languages/pf.js
sha384-YDzioK2VUfo7ZdArDjWEhtgR/zHsogx8SJ9xfToavGK4J4Jq16QZMwo5+mEtHgqL /languages/pf.min.js
sha384-QYdE/O+kYLzPISnj43XU2PBSQGgHxlNW3s9e1CfQn3/mdj2U2lygqPmCk6mh24Ib /languages/pgsql.js
sha384-oJqzx2KYttoB62Br3yGkuDMB7q08+JIjBH1jiTmGfz94kIjSpP8WFYdneQESWp3z /languages/pgsql.min.js
sha384-swGDgtGOmzrsbFAaQRjzvGs0hhe0N86mfHIuisr3W9AT0hiheGyRORSGrbMDGOw5 /languages/php.js
sha384-Xd0AQIkWCEjBL8BNMtDMGVqFXkf445J6PNUJdXbh334ckjeSa90wtSUjXySuz+rt /languages/php.min.js
sha384-2e98DY/C99xqtaiqUR14UMy3FOrroooxZ43kHHlDMdLl45ymAaJ6OgWO1m95lfc5 /languages/php-template.js
sha384-qgAIdznqUzBBAS4nOYdZKnhkSxER8jn7g10EW176MLksxvnQCBcXOdawfqoRb67X /languages/php-template.min.js
sha384-v4qiQbdZu8obdLOFoHrZxA44mmxnjZUelyHe7A6RuqmckxO5weYQVrN8Dx2UpAR1 /languages/plaintext.js
sha384-hE+znpd5xggEBW6IccZoCI0mgFHAfLVuqT/7aW8RakaQ4UJnI058SfIX3lhdGxtE /languages/plaintext.min.js
sha384-74bIBOdSGr097e2nt0jZfkswhU0MM23YhlDNdzyHT+z/eQDI6VCNrpkVobRvMoVz /languages/pony.js
sha384-7llnZwKgcknuqjXcEeUP/boMrzjXxhhfH9afFHqRtqRpnfdGWZJzcAqfq9QHWx5E /languages/pony.min.js
sha384-/AFMsGSWoas65G5mSpnY9M1vnVP/9qhQW9yGZCpgbuL36JxWKX6MABh2hj6pc4yV /languages/powershell.js
sha384-0u0NM3ve01ej9h9zRzZ/ztDGe1h07d6TStpNoJ4f/50I/vtoCsDHI2PfzDZSYz8q /languages/powershell.min.js
sha384-YyOpJjUHKDExvBHAAZLYGW3flcxbgjI3plJ/wzjTYUZut9XfnUjvPJY6saenN05Y /languages/processing.js
sha384-a+nwjMBrHHj4K59nk1SmGnkQ19/zLAQrOQaVUYsiTcBr+HeKqUrjvw5Znx6HQeS7 /languages/processing.min.js
sha384-mnm6RN61wmjIzbB/YIRAChmbfLKezfCSOCAQEe4Z9YsZwtEBxkseWepidk8vxBMX /languages/profile.js
sha384-VSUySmqeUUatYNFPFFZAHEmgpdfedIGFcZyQv/6OIsLc1vR1Wl1SimExn23za0FB /languages/profile.min.js
sha384-XxBy0hHMukr1KPCv2IW7TSLJlY8dy7PvNWmYq4yCjcduXU1gqYWoydV3hbQpSydO /languages/prolog.js
sha384-G0I/5pg6Q0by4a7utLoEKB0+vqUs6FrSfTgIyjVkOCW8I1R5A7R1mDVPpgvHKd0r /languages/prolog.min.js
sha384-QtRG6CHajNu8yO49AKRA40c5f+/TgBUmd4fdF1bUJ2M5YrnNowOBcglGip1EuK/E /languages/properties.js
sha384-sgNTtpv5Paw0Ux4jH4t4kp0frMSOTtws8wtoYOeystt+hVEKHDinMYYW/ZR5SKI9 /languages/properties.min.js
sha384-mBGX8CxN3Neyfl/Zf2gq+OUHJ1YA7PWR7YBBRTN+cILYNp7tI0pzwVMcVH1NBns+ /languages/protobuf.js
sha384-S+PDVP8B6WN9X4KJd8TaIK8Sj6BTdd9QAqCRzIYSuwxAQHpcGbZjHNBVrC9qUfrh /languages/protobuf.min.js
sha384-pOa0jdmHPUu2PfiIIjyrWloPr4N8VIX0rM+g5G9Drb41fuJWRkGvEZqQ0HKRKtqb /languages/puppet.js
sha384-NzOBHXRL0Tvx3y08Z9UFLnfEortdDn+7h0HagnNff40zn4M3PrMQqY6tPYO+APjJ /languages/puppet.min.js
sha384-gC2DE1FVzI4++xVFai3IsVy6Jls+kRC2yaUjpRzt9vXbjfb1cbxUexL48mEA3alQ /languages/purebasic.js
sha384-EXxXI12fO4TsAJfYNsrfXpOeaiw4OyAte9Qly3UTiBavCYLG4t3LEgjP1KzvWs7b /languages/purebasic.min.js
sha384-WNah6F2QDUbmrNCi0fSEyKJbSb01S1ijnoiwbDnegW7dm2Cz/H1CiH1HhWlUvj01 /languages/python.js
sha384-YDj7s2Wf0QEwarV3OB8lvoNJJCH032vOLMDo2HDrYiEpQ+QmKN+e++x3hElX5S+w /languages/python.min.js
sha384-fA29QmwJSF4aooD6d8HMQ/ua3/kQuT8Jim6Uan8SAPd/lMvkeKnzsSj4cS54HVhJ /languages/python-repl.js
sha384-WYz+BIjSZsSWkXPxYY/cN0c3yN3N98f1+JASXaEegpsqQqN0/OpVe9PbiyQ8IFA4 /languages/python-repl.min.js
sha384-APYDLZ6R/GoaBhuK1WZXdKmcXqc0khrcl8CCj29jYjOL5R158pCZXl+eyMNhdLev /languages/q.js
sha384-0wLjaNaUVh/kcNbtLmLZkz9NX8VtSkyPUrRqKAjtf2u056iIzXgTRn1fonvwhvJU /languages/q.min.js
sha384-G6PnZyzJd4CKlp2J1b/mLhMm87AZbdOdEXQLmG5eZqclORmQyExauasU35YjYI/E /languages/qml.js
sha384-mj9QNbYd7ibOAOnrCkY/1jRyke9TPExrkIU7ouV8xtHVmUrM1w/7OoGjiVin6YdB /languages/qml.min.js
sha384-as0oqcSBkBCW8JS5FUngtpwqXH0qK8dlfpPE2IYoERAn+MRsJdlb89n/hgKYshzJ /languages/r.js
sha384-u+x0P8zfV33Hs+FIbrPIw38IUbqKyYA1yEdL911mrwtUJxaNJtMOdB2injOcZMmF /languages/r.min.js
sha384-c0jNHE0LWpUcibYWDzK1zHS3OT6qpIvej1M6FquegEpm9C0zLwORz+QBN3fT3Zoa /languages/reasonml.js
sha384-IdLfiBtws/HD6C/aQaDlw4/xILR8XJV16CBZtTPLs80HNyGsUpNwQmc2+g0aXPhg /languages/reasonml.min.js
sha384-Hwu7t/Tjfl7JEJSV3LoJXYbOX52UyDBYMsWOo3Cwwnu58AmNmo/SniAE/BOLQvmp /languages/rib.js
sha384-Jc59QIC9wiM+tNA9lnx0Cb23+WrXa6PT2LgOAhVPwyx490IJRKdt9/NGT1ZyQc6y /languages/rib.min.js
sha384-tXUWZKRlJiMQMX7YMC8gqUI9tSHFu76aym5JXLpHkJTLuWr55sRM6Ob6VQngsrPF /languages/roboconf.js
sha384-KR6QhCPclzoQUVJZiCDWkDytFrjI7H3z8gCN39ACvRPSTZ0enCcrDTxB33nJ+N8J /languages/roboconf.min.js
sha384-D0MR4/f128phGjx2oxFi3hAhfkfWlDwGetOJWQdimjnbFaWNeb3uLhnh5OeMaDZV /languages/routeros.js
sha384-1lod0Mw40xIsxTmsdIbezbyXhu12CZLPOjFSHJrEiWgJteWmTeSQlluzoW1Sx/aw /languages/routeros.min.js
sha384-ZnEz0w5YKa4T2FMJP08uWJdNC2bZbrL9jpDeH9vJZRKBqJxwO/8Hl1UjzRc3xg3m /languages/rsl.js
sha384-9rzhEAd8EfJCPELcwmTS0OeWys1RBPz26Oghv+u7GT0oXa1PrDPxfsPU7SHQ3fKn /languages/rsl.min.js
sha384-nHCt5/kJd7LUp6BbVLNuUH0zziFVRS2Qqj9whqRpVpgzybI61OZlKQRPv4evHrJO /languages/ruby.js
sha384-LOFdRHZ5u+oZg5Wh8DpkiJQR/w8egcUhJJoo95XVQ846DEwuRxGUujx+tQKyqXme /languages/ruby.min.js
sha384-tKZjVbK0HuRYruP4N2FhizT2r7XX+UwLy/FMluwdOchHiNz2T8Ue6myuy7TmkStf /languages/ruleslanguage.js
sha384-Ydu8edRhwf0Q3mwbCRUPCkhYopMD4JzvyZL6YsOLcDFpQ+7WdKmlBk1jFbCfgfJ6 /languages/ruleslanguage.min.js
sha384-rCXn7K5j/lD2PrSex2XCqyLKap2Ibnsls0uQCx4ZaezI1FJ5RYvoWcsAl/v8SKlE /languages/rust.js
sha384-VTNxHMz2AmpHxzSm8SvRI0As5+wID2j2XJBFtWTic2iEK8WbXgR1fymVQS9S2DvY /languages/rust.min.js
sha384-SbgmLFWA9CPCFnT3Chy1w6SZjYluPvLcMjlzlP1qkuoRhTvblhBo3J+8OD8iIYEn /languages/sas.js
sha384-Jf3WKZhkMK2CEuMszIpp3i7w+QbmgOIzDd7ueAeHTH5GXOTZK38bHB9f1JZDETC7 /languages/sas.min.js
sha384-/YpJjx5RnzUuJcVaJYIEU44EkoCD/KX5CiVM8Aj7wfkZj6EYw9g3CPauWCDI+ljG /languages/scala.js
sha384-SA6LxB7n4Y1G9cv1TTU9P9QCpQXJaNURCzOyoGjV/D9gkhcHYSsYERw8IVffZ8uD /languages/scala.min.js
sha384-ignlYk/xCspDyEW50tPeaOucE/nrcQggQ3bcN2i+B+wLxIfIIQ+N97yOztUtbyfS /languages/scheme.js
sha384-0BOfMS8PT03WW7dkaI3LgBx21PwSLCxWucHG9qTsAXRKMLHcIMHuu9spz3606P3o /languages/scheme.min.js
sha384-UPrZlXQxfHdLMm3sK7qCNGz4OayerTRa8E0IPpgdW4xZbHYYmZPrbJymBGmTB9pj /languages/scilab.js
sha384-7qPMH09XA8AJjij3fvK4DR/tZf1ajIf1EEmvQfpXkGKeC8tAqz+TqdavHtkxKHet /languages/scilab.min.js
sha384-1MNX8RDXroN7D+Jeq9f6EysUNAuhuQMq32wKcGB5yEDEJalUxEoSjfRVJhF0+YV6 /languages/scss.js
sha384-7CdPzO1jMDolQK+LCRlGiW1Oka6fSiQdIiqfTvHeBk2Hm2iYNoqMn60aJ9HdKkjb /languages/scss.min.js
sha384-KYOeDvyFo8fJObDV1L1aoPnfs6XG68LL6j3INM7McXyRYtBZF7DdUsNjK25dtxKo /languages/shell.js
sha384-olAuUjfRvTi/iEH4RXRpaq/G1iJGizn7OefkyJLQYuqNhh1xAV5dnUrkH/LlPd9j /languages/shell.min.js
sha384-skMR3kOnYlfe2aXWnzlGUGIc4zVPi3p1QHI6bGl8pfoOkA+3ZvkLC4/2ZZvXgWGs /languages/smali.js
sha384-Lmk7Z08oieDOChjC++YInGivPR7XvfTJFXunqImOsFq71wdP7fqHZs4p2E68NW3z /languages/smali.min.js
sha384-NuLgWCPZ3g3KflwSf+QirMxw1ElIvZ+J7ZHMLQFyatG5dfBZwXk37xA1TN/jQBtH /languages/smalltalk.js
sha384-9xPGl0IyZCFsMCTnDoxktyJm/1P4qtRk+WCscFY2JCkrXgrZv1TYV8tw/iLVgQeH /languages/smalltalk.min.js
sha384-jEDC5tnQCMVvN5T/xpdyU6LpHskzNm777zheXkyV2I3IGGr8LOWaa5rl59OdkJ3C /languages/sml.js
sha384-Ahqruee4TcqHPxJ0qWMd8AwCzxn32ztYC5BsgO5a+9StIlucQfHfbxf3shJ4knjC /languages/sml.min.js
sha384-bWnZnSt7i89crfq/aRzwwRlA+NRHyOO7+PWc2y2KoD9G80bh8rWaWAtZ0w7nHqKb /languages/sqf.js
sha384-9ckCCmGBhiko6gWuy8HUDGhwBCAW7OuTPEPZcHh5qfpr78oz2zfuU5svTcxdiYvC /languages/sqf.min.js
sha384-w/OmtgUvmlKWaVatpcvuEQxP7bkJzI5gLeeQkuXjApJNiQvNmXFL2PBM5RWgKqDr /languages/sql.js
sha384-2uzCjI3OPwJce6i2hphsYs1qqTqRrDnfPXbfjZggPWy2/Lgl8gzV9Hyl0jhhoWy4 /languages/sql.min.js
sha384-9BHbk0zbyMhgpT1FDZTkYC4YgFeFV+DmjHNfBqriy9yjlWLZNnc3E27zRwIZlnSN /languages/stan.js
sha384-7k+vuglPXrNadgl5jySkKUUjOjk6NsVV9+M+JCRp3wP++kZjRK2hogRJrGBrsURm /languages/stan.min.js
sha384-6wTaFaOf/MgeYXd+9i+gTHuhhU1m9mqM1bBmmX4ehatAUcVEQF/dQMncS7dD9SnD /languages/stata.js
sha384-b+wzPiDP5nqHGvkTkMg9KIp7xjHmb+1LGAlpH21g+Oyksvq3NHknM39ZLdZ8RdVE /languages/stata.min.js
sha384-tB+/97g3o9due/vEwbAp14kmXaqV4lHlUOMQLL85pxKFT0brrdFzAw4GRFTSMNUP /languages/step21.js
sha384-VOaOLJhjy62zrnq3N6nBKbU8yCsDVEZI+l6Ch11xdI8FuUHFqU2AT7FX3TDwKChO /languages/step21.min.js
sha384-0Yy/cZlbG78zeysOAMDd9OMKj8FrGaVfg/IH9zKh0JCBGMW0gabvTIAtdif4HpSh /languages/stylus.js
sha384-MoaYmkURrT/giCrgWXgBEOOXP+nEFtMKjRY2XIzWehdGT2GN73w+dTvyP+jZhEZL /languages/stylus.min.js
sha384-3Lq7e8rtXKz/bA1fYeAuqLxtx03wkycboNfzHT7akhuZa+NqA7nk9CW8LFWzN4gj /languages/subunit.js
sha384-OYfRTEinJxuatPmhe9R3rJ+lG6jNXUHvSsBpVDDqU+RWJZ6wjkD+QYuM7dBnUbG6 /languages/subunit.min.js
sha384-avfxrZ7nwXDWWFaYzhYylhlr8UCb5MOAoBoIxEYvDmPl2iPfA/4qk4jrIYdyGVA0 /languages/swift.js
sha384-TfALNLT6HJzZieazgsVvFM0DzFWQsgl0d7mdwPLyg1yg7XE4QwLY4jqmJRNnI1S4 /languages/swift.min.js
sha384-bMntl9I1jIv7DactTrFrnf4UXjmozRVAyNrqBMQRg9oYFBAfXVxEAqfW3E6N+2tl /languages/taggerscript.js
sha384-A6Duvm756b9xHe0gOvjVWitUxu8PLX7g461lNhVaPtpoSJ7GyUNdLCD+OBRj63Ue /languages/taggerscript.min.js
sha384-iCcmD7/12YWwDRo9T5p2LIAhmm023UAiKAsQFKjHRwiS9IvktR0X8k3nlhhS5f4R /languages/tap.js
sha384-39o8THRvVOl9DEzvSrCMsUCub41ibGqjj/O73e4Mz08I6fmgUSfHplFTJkBqxKz1 /languages/tap.min.js
sha384-OAtXQr6xPAIGXvXmnCu7EHg1sTR+wCzucocvL/+DT76XmauXNX82f6F4k9lc6wQe /languages/tcl.js
sha384-r9OPcD/aFYhqKM2SiJ9rWMahBytgRLrbYvJ77m1AaO3ghVCWBSK7f1hWyc3HuWX1 /languages/tcl.min.js
sha384-SOTB/lvlCeAxebj328FS4563n/qKmAsk88ti6oeMZQgn4n+h3McaDCbWbwFx62MN /languages/thrift.js
sha384-mLeI0s1+uHK7HpID/RvvgQ1QGBKOce7RvWMnxBDy/boNuW0W9JcaiaT+0vOpdMy7 /languages/thrift.min.js
sha384-qKuXdiOcajhtJwcOQy/rQ8Km4JRU52M3HUNffbw0NrsLVOcSV9fTYBOVYb644e7t /languages/tp.js
sha384-qy3fDByfNQpIPnTx39nNL76o4ofcek+QnbxyLOWY1qgtRXq/TYz2vHv59LWC8i/e /languages/tp.min.js
sha384-/hnpdZfdi8MEHBYn7ZqWrgiNsAMlBh5hruT/l9qD/wFozJgEBn+MSiZVNkpAzNiM /languages/twig.js
sha384-XlDRKuo3Ea9pIqT0CUrf4Vp4ZN/AItGl/aUiL/4AgHSC2N7xJI1commJxYZo0I1T /languages/twig.min.js
sha384-4q0Mj1AHSvVdgi6nXDGdkiHZQcme/PcCE+MvwCvnAIZSjhJfk3UpjJU2nn2eImWz /languages/typescript.js
sha384-rfwxAwuWzb2XdSU7HN3IhrSyCq96Nj4p1ZYPCNAGbqtnPsaWl8d5eSypxPbW6alT /languages/typescript.min.js
sha384-eZUT2M623ineqATk9gMm7Iio8SiQu+9+MlCM4WvKccHhFGoThFLfa0sbnGPz/5e1 /languages/vala.js
sha384-qsiVnf7Cwh6dClTlwLWKcQ/3IY15+qMaVWIDEliKyO80Cpb7z/ie9etw4fo0ydyo /languages/vala.min.js
sha384-ZtGEPec3UETo0w1s5T3Xy0jf1B3d/QJXncovO627I/9j94UdhtJ8N2NFu8vn6p6H /languages/vbnet.js
sha384-cHdAVgva6ADwmJQeO9JPevHoX/8tL0MEqz3vMAxu+Jy6X3m1TJ+U9qYeb3E1Rny0 /languages/vbnet.min.js
sha384-OAqGGQAYPS6NxSI7WIWkCBd0IGQD3bsBElVnokVg7252qtGTVPFjk6LHG1Iqwbe0 /languages/vbscript.js
sha384-dO0iYHNUbREvGXvDcxZsIcqf9DCTUASjhhyStXtJFhj9dipN3Ypc94KaHXIQQN24 /languages/vbscript.min.js
sha384-R2QKmi7N0DVttqhNpU5wkftWRTVN+FlOnv18+Bk/zxM/V8qkRBU4yUCpsr9AdK+x /languages/vbscript-html.js
sha384-q0sJ1YFiPW+i+XVme9b3M6adegU2pnXPpwhLvGukidQZIkV9QIzBFdSCkf4LjQAf /languages/vbscript-html.min.js
sha384-1JXyk4uCVxAfcJRwRfNnpt8moDWzGVnTnXk4ucnBTXFoDyeEP1/FqjQCqgutU9nc /languages/verilog.js
sha384-i+ovBPuX6/6IYKo/hvzjiDeWmFZGjQ8GcN5P2JlCvsQQz7tmGhwFBZ7L8O5j6Uih /languages/verilog.min.js
sha384-V2Oynheqj/yMT38G4zcskFIWTJwExjl3g3gvD2a0rga9jbrWA1jdlARfXwdxDFeN /languages/vhdl.js
sha384-lRfKmfeanfEEsne7ndto0E7FFUaI9By4vEr1RZkq+pVhHu4dRKebAURBKne7Dzdm /languages/vhdl.min.js
sha384-6IMlY3eILHj9Dec4eMLmEQNsQF49OiTeVVMNIy1A+PxuMdmoeZ2AnVN/7FEpsgAx /languages/vim.js
sha384-vfNnhhcZoO470LVTMwy7VO8IGy2qRDfw+NOnU7pdz/V8y2uNfjq5Ea7Gljj0dZEn /languages/vim.min.js
sha384-7+RHZQJsgiohPfYs4xGnVLR6bKm1cnz7rvB7mXEb2btmiWhSNq1cuTJsQWK9Ut8w /languages/wasm.js
sha384-TTxGby06lDIEB1Y6l2/Hd2VJbewWQSCESXUcDWC24gJN0IiptruI6DE1l5+w4Qzh /languages/wasm.min.js
sha384-ajpkS/9YOf+vslZQTzqreKjqujPN08Qk5CCsmdoO50DimjJmUcufyrymOt5ZxoLY /languages/wren.js
sha384-6R+wTNBHlaPfbkMRDtjTf5quccaGcE9dSnW1yvuCyvVQWhUxAnDoMfcW5iPuVAZv /languages/wren.min.js
sha384-9s/ZvqYSxQS2hJrT3LuFUwDyxlWEELd50mRuGXt0zzSwafBqEWPsv9xxNsZUF2W4 /languages/x86asm.js
sha384-cw6YAmvOjrczJAb9NkjJRzsXso0VEJrJsLZl8G49s9vVlIFVwG8cR6K8yvpDlGn5 /languages/x86asm.min.js
sha384-MRD7oo7j9xs88dYNM3bGW3ai6Oi9ciSlJv1Mpj+eesS2OX+hYFoiv5ug5p+slMxb /languages/xl.js
sha384-9kjy3NF6jS9dLGRFPKl8hlTtPEwFGUkscHqovyfsjFqblpaGB05HpBzFgHD8Fj8c /languages/xl.min.js
sha384-QAL2h4IMgQaJUJjUy0dSWdAut7o/A272ai8qOsJ8SSm9KMxkdLgH7oGfLGft/EJ0 /languages/xml.js
sha384-CN3No+n1UZXCFYyl+ge5yAPGTNGuH23BdIsFJxntDmEYL94AmoZlNBHGSdjVSjKG /languages/xml.min.js
sha384-7aJnr22Yq+e5qwa1KWt7uJtKR8dDq5WhP6qJ69sAVEAAGSUakK8AxjiiEkaBHimk /languages/xquery.js
sha384-L+i0yBdAaDJKM0LjWrOba9BR58bmNU4O1e7339zfF3ZW0CjnaTJsQ6AuK22TNIVc /languages/xquery.min.js
sha384-3KIoWvJ5JGRH35WAkzreEebY8sug+ZWeaOPS2r1KIfznEU9TtPFpxX6sIgtaiA9G /languages/yaml.js
sha384-bMkvdnz+wPu1ro0fqO3BaDWztc7RzSvw05MQFP6bhJKDcwpkrFYTfTFI9ndkP11l /languages/yaml.min.js
sha384-iw5vfj3aLmmBI0eTXaW58+7oltnHpXTNTSuYj5icFfnsSPIOWXhhIJ54PHmXduA8 /languages/zephir.js
sha384-7A2HawNmptSPS479EB34q/+e8WPft+1zFF1OETV5ZYY/JzzUkfkkqXxng6jV4wB8 /languages/zephir.min.js
sha384-dfob6Kflwhv/Dp44Mc1VK5/eJaNrKMLEo6LRRDFATzaaMoHaCHByLwtyhi0PuP3n /highlight.js
sha384-hIY/n60mwq+8V5Dbi2lkNNWjrtVjaF4ZudRZzcXlxWDuJjPTZhRnIe87YWqTA2Fk /highlight.min.js
```

