<?php
/**
 * @package    Fields - WT Layout select
 * @version    1.0.0
 * @Author     Sergey Tolkachyov, https://web-tolk.ru
 * @copyright  Copyright (C) 2026 Sergey Tolkachyov
 * @license    GNU/GPL http://www.gnu.org/licenses/gpl-3.0.html
 * @since  v.1.0.0
 */

declare(strict_types=1);

namespace Joomla\Plugin\Fields\Wtlayoutselect\Support;

\defined('_JEXEC') or die;

/**
 * Provides reusable normalization helpers for layout and folder paths.
 *
 * @since  v.1.0.0
 */
final class LayoutPathHelper
{
    /**
     * Normalizes folder path to a predictable format.
     *
     * @param   string  $path  Raw folder path.
     *
     * @return  string
     *
 * @since  v.1.0.0
     */
    public static function normalizeFolderPath(string $path): string
    {
        $path = trim(str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $path));
        $path = preg_replace('#' . preg_quote(DIRECTORY_SEPARATOR, '#') . '+#', DIRECTORY_SEPARATOR, $path) ?: $path;

        if (preg_match('#^[A-Za-z]:[\\\\/]#', $path) === 1) {
            return rtrim($path, DIRECTORY_SEPARATOR);
        }

        return trim(ltrim($path, DIRECTORY_SEPARATOR), DIRECTORY_SEPARATOR);
    }

    /**
     * Normalizes layout filename without extension to a relative path.
     *
     * @param   string  $layout  Layout path.
     *
     * @return  string
     *
 * @since  v.1.0.0
     */
    public static function normalizeLayoutName(string $layout): string
    {
        $layout = trim(str_replace(['/', '\\'], DIRECTORY_SEPARATOR, $layout));
        $layout = preg_replace('#' . preg_quote(DIRECTORY_SEPARATOR, '#') . '+#', DIRECTORY_SEPARATOR, $layout) ?: $layout;
        $layout = preg_replace('/\.php$/i', '', $layout) ?: $layout;

        return trim(trim($layout, DIRECTORY_SEPARATOR), '.');
    }

    /**
     * Normalizes legacy non-JSON values into a dot-separated path.
     *
     * @param   string  $value  Stored legacy value.
     *
     * @return  string
     *
 * @since  v.1.0.0
     */
    public static function normalizeLegacyDotPath(string $value): string
    {
        $value = trim($value);
        $value = preg_replace('/\.php$/i', '', $value) ?: $value;
        $value = str_replace(['\\', '/'], '.', $value);
        $value = preg_replace('/\.+/', '.', $value) ?: $value;

        return trim($value, '.');
    }

    /**
     * Creates full dot-separated path from normalized parts.
     *
     * @param   string  $basePath  Base path.
     * @param   string  $layout    Layout name.
     *
     * @return  string
     *
 * @since  v.1.0.0
     */
    public static function composeDotPath(string $basePath, string $layout): string
    {
        if (empty($basePath) || empty($layout)) {
            return '';
        }

        $baseDot = str_replace(['/', '\\'], '.', $basePath);
        $layoutDot = str_replace(['/', '\\'], '.', $layout);

        return trim($baseDot . '.' . $layoutDot, '.');
    }

    /**
     * Validates that folder path is relative and does not contain traversal.
     *
     * @param   string  $folder  Normalized folder path.
     *
     * @return  bool
     *
 * @since  v.1.0.0
     */
    public static function isSafeRelativeFolder(string $folder): bool
    {
        if (empty($folder)) {
            return false;
        }

        if (str_starts_with($folder, '/') || preg_match('/^[A-Za-z]:\//', $folder) === 1) {
            return false;
        }

        foreach (preg_split('#[\\\\/]#', $folder) ?: [] as $segment) {
            if (empty($segment) || $segment === '.') {
                continue;
            }

            if ($segment === '..') {
                return false;
            }
        }

        return true;
    }
}

