<?php
/**
 * @package    Fields - WT Layout select
 * @version    1.0.1
 * @Author     Sergey Tolkachyov, https://web-tolk.ru
 * @copyright  Copyright (C) 2026 Sergey Tolkachyov
 * @license    GNU/GPL http://www.gnu.org/licenses/gpl-3.0.html
 * @since  v.1.0.0
 */

declare(strict_types=1);

namespace Joomla\Plugin\Fields\Wtlayoutselect\Field;

use Joomla\CMS\Factory;
use Joomla\Filesystem\Folder;
use Joomla\CMS\Form\Field\GroupedlistField;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\Database\DatabaseInterface;
use Joomla\Plugin\Fields\Wtlayoutselect\Support\LayoutPathHelper;

\defined('_JEXEC') or die;

final class WtlayoutselectField extends GroupedlistField
{
    /**
     * @var string
     * @since  v.1.0.0
 */
    protected $type = 'wtlayoutselect';

    /**
     * @var array<int, array{element: string, name: string}>|null
     * @since  v.1.0.0
 */
    private ?array $siteTemplatesCache = null;

    /**
     * Renders grouped list with escaped option values.
     *
     * Core groupedlist layout uses `option.key.toHtml = false`, which breaks JSON in value attribute.
     *
     * @return  string
     * @since  v.1.0.0
 */
    protected function getInput(): string
    {
        $data = $this->collectLayoutData();
        $data['groups'] = (array) $this->getGroups();

        $html = [];
        $attr = '';

        $attr .= !empty($data['class']) ? ' class="form-select ' . $data['class'] . '"' : ' class="form-select"';
        $attr .= !empty($data['size']) ? ' size="' . (int) $data['size'] . '"' : '';
        $attr .= !empty($data['multiple']) ? ' multiple' : '';
        $attr .= !empty($data['required']) ? ' required' : '';
        $attr .= !empty($data['autofocus']) ? ' autofocus' : '';
        $attr .= (string) ($data['dataAttribute'] ?? '');

        if (!empty($data['readonly']) || !empty($data['disabled'])) {
            $attr .= ' disabled="disabled"';
        }

        if (!empty($data['onchange'])) {
            $attr .= ' onchange="' . $data['onchange'] . '"';
        }

        $options = [
            'list.attr'          => $attr,
            'id'                 => (string) $data['id'],
            'list.select'        => $data['value'],
            'group.items'        => null,
            'option.key.toHtml'  => true,
            'option.text.toHtml' => false,
        ];

        if (!empty($data['readonly'])) {
            $html[] = HTMLHelper::_('select.groupedlist', $data['groups'], null, $options);

            if (!empty($data['multiple']) && is_array($data['value'])) {
                $values = $data['value'];

                if (count($values) === 0) {
                    $values[] = '';
                }

                foreach ($values as $value) {
                    $html[] = '<input type="hidden" name="' . $data['name'] . '" value="' . htmlspecialchars((string) $value, ENT_COMPAT, 'UTF-8') . '">';
                }
            } else {
                $html[] = '<input type="hidden" name="' . $data['name'] . '" value="' . htmlspecialchars((string) $data['value'], ENT_COMPAT, 'UTF-8') . '">';
            }
        } else {
            $html[] = HTMLHelper::_('select.groupedlist', $data['groups'], (string) $data['name'], $options);
        }

        return implode($html);
    }

    /**
     * Returns grouped options from configured layout folders.
     *
     * Behavior is aligned with ModulelayoutField:
     * - Base group per configured folder.
     * - Override groups per site template.
     * - Base entries overridden by templates are hidden from base group.
     *
     * @return  array<string, array<int, object>>
     * @since  v.1.0.0
 */
    protected function getGroups(): array
    {
        $groups = parent::getGroups();

        foreach ($this->parseFolders((string) $this->element['folders']) as $folder) {
            foreach ($this->collectGroupsForFolder($folder) as $groupName => $items) {
                if ($items === []) {
                    continue;
                }

                if (!isset($groups[$groupName])) {
                    $groups[$groupName] = [];
                }

                foreach ($items as $value => $label) {
                    $groups[$groupName][] = HTMLHelper::_('select.option', $value, $label);
                }
            }
        }

        if ($this->isAddEmptyDefaultOptionEnabled()) {
            $defaultOption = HTMLHelper::_(
                'select.option',
                '',
                Text::_('PLG_FIELDS_WTLAYOUTSELECT_OPTION_DEFAULT')
            );
            $groups = array_merge([[$defaultOption]], $groups);
        }

        return $groups;
    }

    /**
     * @return  bool
     * @since  v.1.0.0
 */
    private function isAddEmptyDefaultOptionEnabled(): bool
    {
        $raw = trim((string) $this->element['add_empty_default_option']);

        if ($raw === '') {
            return true;
        }

        return !in_array(strtolower($raw), ['0', 'false', 'no', 'off'], true);
    }

    /**
     * @param   string  $folder  Configured folder path.
     *
     * @return  array<string, array<string, string>>
     * @since  v.1.0.0
 */
    private function collectGroupsForFolder(string $folder): array
    {
        $normalizedFolder = LayoutPathHelper::normalizeFolderPath($folder);

        if (empty($normalizedFolder)) {
            return [];
        }

        $baseGroupName = $this->formatGroupName($normalizedFolder);

        $baseEntries = [];
        $basePath = $this->resolveBasePath($normalizedFolder);

        if ($basePath !== null) {
            $baseEntries = $this->scanLayoutEntries($basePath, $normalizedFolder);
        }

        $templateEntriesByTemplate = [];

        foreach ($this->getSiteTemplates() as $template) {
            $templatePath = $this->resolveTemplateOverridePath($template['element'], $normalizedFolder);

            if ($templatePath === null) {
                continue;
            }

            $entries = $this->scanLayoutEntries($templatePath, $normalizedFolder);

            if ($entries === []) {
                continue;
            }

            $templateEntriesByTemplate[] = [
                'name'    => $template['name'],
                'entries' => $entries,
            ];
        }

        $overriddenLayouts = [];

        foreach ($templateEntriesByTemplate as $templateData) {
            foreach ($templateData['entries'] as $layout => $_entry) {
                $overriddenLayouts[$layout] = true;
            }
        }

        $groups = [];
        $baseItems = [];

        foreach ($baseEntries as $layout => $entry) {
            if (isset($overriddenLayouts[$layout])) {
                continue;
            }

            $baseItems[$entry['value']] = $entry['label'];
        }

        if ($baseItems !== []) {
            ksort($baseItems);
            $groups[$baseGroupName] = $baseItems;
        }

        foreach ($templateEntriesByTemplate as $templateData) {
            $templateItems = [];

            foreach ($templateData['entries'] as $entry) {
                $templateItems[$entry['value']] = $entry['label'];
            }

            if ($templateItems === []) {
                continue;
            }

            ksort($templateItems);
            $groups[$baseGroupName . ' -> ' . $templateData['name']] = $templateItems;
        }

        return $groups;
    }

    /**
     * @param   string  $templateElement  Template element name.
     * @param   string  $folder           Configured folder path.
     *
     * @return  string|null
     * @since  v.1.0.0
 */
    private function resolveTemplateOverridePath(string $templateElement, string $folder): ?string
    {
        if (empty($templateElement)) {
            return null;
        }

        $override = JPATH_SITE . DIRECTORY_SEPARATOR . 'templates' . DIRECTORY_SEPARATOR . $templateElement . DIRECTORY_SEPARATOR . 'html' . DIRECTORY_SEPARATOR . $folder;

        if (!is_dir($override)) {
            return null;
        }

        $real = realpath($override);

        return $real !== false ? str_replace('\\', '/', rtrim($real, '/\\')) : str_replace('\\', '/', rtrim($override, '/\\'));
    }

    /**
     * Returns enabled site templates.
     *
     * @return  array<int, array{element: string, name: string}>
     * @since  v.1.0.0
 */
    private function getSiteTemplates(): array
    {
        if ($this->siteTemplatesCache !== null) {
            return $this->siteTemplatesCache;
        }

        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true)
            ->select([$db->quoteName('element'), $db->quoteName('name')])
            ->from($db->quoteName('#__extensions'))
            ->where($db->quoteName('client_id') . ' = 0')
            ->where($db->quoteName('type') . ' = ' . $db->quote('template'))
            ->where($db->quoteName('enabled') . ' = 1')
            ->order($db->quoteName('name') . ' ASC');

        $db->setQuery($query);
        $rows = $db->loadAssocList();

        if (!is_array($rows)) {
            $this->siteTemplatesCache = [];

            return $this->siteTemplatesCache;
        }

        $templates = [];

        foreach ($rows as $row) {
            $element = isset($row['element']) && is_string($row['element']) ? trim($row['element']) : '';
            $name = isset($row['name']) && is_string($row['name']) ? trim($row['name']) : $element;

            if (empty($element)) {
                continue;
            }

            $templates[] = [
                'element' => $element,
                'name'    => !empty($name) ? $name : $element,
            ];
        }

        $this->siteTemplatesCache = $templates;

        return $this->siteTemplatesCache;
    }

    /**
     * @param   string  $folder  Folder path relative to Joomla root or absolute path.
     *
     * @return  string|null
     * @since  v.1.0.0
 */
    private function resolveBasePath(string $folder): ?string
    {
        $folder = LayoutPathHelper::normalizeFolderPath($folder);

        if (!LayoutPathHelper::isSafeRelativeFolder($folder)) {
            return null;
        }

        $absolute = JPATH_ROOT . DIRECTORY_SEPARATOR . $folder;
        $realPath = realpath($absolute);
        $rootPath = realpath(JPATH_ROOT);

        if ($realPath === false || $rootPath === false || !is_dir($realPath)) {
            return null;
        }

        $normalizedRealPath = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, rtrim($realPath, '/\\'));
        $normalizedRootPath = str_replace(['/', '\\'], DIRECTORY_SEPARATOR, rtrim($rootPath, '/\\'));

        if (!str_starts_with($normalizedRealPath . DIRECTORY_SEPARATOR, $normalizedRootPath . DIRECTORY_SEPARATOR)) {
            return null;
        }

        return $normalizedRealPath;
    }

    /**
     * Scans directory and returns entries keyed by relative layout id.
     *
     * @param   string  $directory  Absolute directory path.
     * @param   string  $prefix     Configured base path.
     *
     * @return  array<string, array{value: string, label: string}>
     * @since  v.1.0.0
 */
    private function scanLayoutEntries(string $directory, string $prefix): array
    {
        $entries = [];
        $baseLength = strlen(rtrim(str_replace('\\', '/', $directory), '/')) + 1;
        $normalizedPrefix = LayoutPathHelper::normalizeFolderPath($prefix);

        foreach (Folder::files($directory, '\\.php$', true, true) as $filePath) {
            $normalized = str_replace('\\', '/', $filePath);
            $relative = substr($normalized, $baseLength);

            if ($relative === false || empty($relative)) {
                continue;
            }

            $relativeNoExt = preg_replace('/\\.php$/', '', $relative);

            if (!is_string($relativeNoExt) || empty($relativeNoExt)) {
                continue;
            }

            $layoutName = LayoutPathHelper::normalizeLayoutName($relativeNoExt);
            $rawValue = $this->encodeRawValue($normalizedPrefix, $layoutName);

            if ($rawValue === null || empty($layoutName)) {
                continue;
            }

            $entries[$layoutName] = [
                'value' => $rawValue,
                'label' => str_replace('\\', '/', $relative),
            ];
        }

        return $entries;
    }

    /**
     * @param   string  $rawFolders  User-defined folders separated by commas or lines.
     *
     * @return  array<int, string>
     * @since  v.1.0.0
 */
    private function parseFolders(string $rawFolders): array
    {
        $parts = preg_split('/[\r\n,;]+/', $rawFolders) ?: [];
        $folders = [];

        foreach ($parts as $part) {
            $folder = LayoutPathHelper::normalizeFolderPath($part);

            if (empty($folder)) {
                continue;
            }

            $folders[] = $folder;
        }

        return array_values(array_unique($folders));
    }

    /**
     * Formats group name from the configured base path.
     *
     * If the path depth is more than 3 levels, it is compacted to:
     * first/.../penultimate/last
     *
     * @param   string  $folder  The configured base path.
     *
     * @return  string
     * @since  v.1.0.0
 */
    private function formatGroupName(string $folder): string
    {
        $normalized = LayoutPathHelper::normalizeFolderPath($folder);
        $parts = array_values(array_filter(preg_split('#[\\\\/]#', $normalized) ?: [], static fn(string $part): bool => !empty($part)));

        if ($parts === []) {
            return $normalized;
        }

        if (count($parts) > 3) {
            return $parts[0] . '/.../' . $parts[count($parts) - 2] . '/' . $parts[count($parts) - 1];
        }

        return implode('/', $parts);
    }

    /**
     * Encodes the persisted raw value as JSON.
     *
     * @param   string  $basePath  Base path prefix.
     * @param   string  $layout    Layout name relative to the base path.
     *
     * @return  string|null
     * @since  v.1.0.0
 */
    private function encodeRawValue(string $basePath, string $layout): ?string
    {
        if (empty($basePath) || empty($layout)) {
            return null;
        }

        $encoded = json_encode(
            [
                'basePath' => $basePath,
                'layout'   => $layout,
            ],
            JSON_UNESCAPED_SLASHES
        );

        return is_string($encoded) ? $encoded : null;
    }
}


